/*
  This file contains classes master, slave
  and top level netlist for simple TL1 point-to-point demo
*/


#define DEBUG

// Remove comment to simulate an asynchronous slave
//#define ASYNCRONOUS_SLAVE 

#include <systemc.h>
#include "ocp_tl1_globals.h"
#include "ocp_tl1_data_cl.h"
#include "tl_channel.h"  



// Copyright (c 2003 OCP-IP
// Simple non-pipelined TL1 master

#include "tl_master_if.h"

SC_MODULE(master) {
  
  // Port definitions
  sc_port<TLmasterIF<OCP_TL1_DataCl<int, int> > > ocp;
  sc_in_clk  clk;

  void request_thread () {
    bool read_cycle = 0;
    bool read_done = 1;
    wdata = 0;
    
    while (1) {
      wait(); // Wait for positive clock edge
      if (!read_cycle) {
	read_cycle = 1; // Toggle write and read cycles
	if (!ocp->MgetSbusy()) { // RTL equivalent: SCmdAccept='1'
	  ocp->MputWriteRequest(); // RTL: MCmd <= OCP_WRITE
	  ocp->GetDataCl()->MputMData(wdata); // RTL: MData <= wdata;
#ifdef DEBUG
	  cout << "Master puts write request";
	  cout << ", data = " << wdata;
	  cout << "  at time  = " << sc_time_stamp().to_seconds() << endl;
#endif
	  wdata++;
	}
      }
      else {
	if (read_done) {
	  if (!ocp->MgetSbusy()) {
	    read_done = 0;
	    ocp->MputReadRequest(); // RTL: MCmd <= OCP_READ
#ifdef DEBUG
	    cout << "Master puts read request";
	    cout << " at time  = " << sc_time_stamp().to_seconds() << endl;
#endif
	  }
	}
	else {
	  if (ocp->MgetResponse(0)) { // RTL: SResp = not OCP_IDLE; 
	    rdata = ocp->GetDataCl()->MgetSData(); // RTL: rdata <= SData
	    ocp->Mrelease(); // RTL: MRespAccept <= '1'
	    read_done = 1;
	    read_cycle = 0; // Toggle write and read cycles
#ifdef DEBUG
	    cout << "Master gets read response";
	    cout << ", data = " << rdata;
	    cout << " at time  = " << sc_time_stamp().to_seconds() << endl;
#endif
	  }
	}
      }
    }
  }
      
  SC_CTOR(master) {
    SC_THREAD(request_thread);
    sensitive_pos(clk);
  }

  int wdata, rdata;

};



// Copyright (c 2003 OCP-IP
// Simple non-pipelined TL1 slave

#include "tl_slave_if.h"

SC_MODULE(slave) {

  // Port definitions
  sc_port<TLslaveIF<OCP_TL1_DataCl<int, int> > > ocp;
  sc_in_clk clk;

  // Globals
  int data;
  
  void response_thread () {
    bool wr;
    bool tmp;
    bool response_pending = 0;

    while (true) {
      wait(); // Wait for clock edge or channel event (see constructor)

      if (!response_pending) {
	tmp = ocp->SgetRequest(1); // RTL equivalent: if (MCmd='1') then SCmdAccept='1'
	if (tmp) {
	  wr = ocp->IsWrite(); // Test read or write
	  if (wr) {
	    data = ocp->GetDataCl()->SgetMData(); // RTL: data <= MData
#ifdef DEBUG
	    cout << "Slave gets write request";
	    cout << ", data = " << data;
	    cout << "  at time  = " << sc_time_stamp().to_seconds() << endl;
#endif
	  }
	  else {
	    response_pending = 1;
#ifdef DEBUG
	    cout << "Slave gets read request";
	    cout << "  at time  = " << sc_time_stamp().to_seconds() << endl;
#endif
	  }
	}
      }
      if (response_pending) { // Same cycle response
	tmp = ocp->SputResponse(); 
	if (tmp) {
	  ocp->GetDataCl()->SputSData(data);  // RTL: SData = data
	  response_pending = 0;
#ifdef DEBUG
	  cout << "Slave puts read response ";
	  cout << ", data = " << data;
	  cout << "  at time  = " << sc_time_stamp().to_seconds() << endl;
#endif
	}
      }
    }
  }

  SC_CTOR(slave) {
    SC_THREAD(response_thread);
#ifdef ASYNCRONOUS_SLAVE
    sensitive(ocp); // for asynchronous slave, like aSRAM
#else
    sensitive_pos(clk); // for synchronous slave, like sSRAM
#endif
  }
};




// Copyright (c 2003 OCP-IP
// Top level netlist for simple TL1 example

int sc_main(int, char*[])
{

  bool sync = true;
  TL_Channel<OCP_TL1_DataCl<int, int> >  ch0("ch0", sync);

  slave sl1("sl1");
  master ms1("ms1");

  sc_clock clk("clock", 1000, SC_MS);

  ms1.ocp(ch0);
  ms1.clk(clk);

  sl1.ocp(ch0);
  sl1.clk(clk);

  sc_start(20000, SC_MS);

  return(0);
}
