//////////////////////////////////////////////////////////
//
//  (c) Copyright OCP-IP 2003
//  OCP-IP Confidential and Proprietary
//
//  Structures to pass data to and from the OCP Channel
//  for both the TL1 and TL2 OCP specific channels.
//
// Joe Chou, Sonics Inc., joechou@sonicsinc.com
// Alan Kamas, for Sonics Inc., aok@sonicsinc.com, www.kamas.com
// Yann Bajot, Prosilog, bajot@prosilog.com
//
//  October 1st, 2003
//////////////////////////////////////////////////////////

#ifndef _OCP_GLOBALS_H
#define _OCP_GLOBALS_H

#include <string>
#include <assert.h>

// needed for gcc 3.3:
using namespace std;

// -------------------------------------------------
// OCP 2.0
// -------------------------------------------------

// MCmd encoding
enum OCPMCmdType {
    OCP_MCMD_IDLE = 0,
    OCP_MCMD_WR,
    OCP_MCMD_RD,
    OCP_MCMD_RDEX,
    OCP_MCMD_RDL,
    OCP_MCMD_WRNP,
    OCP_MCMD_WRC,
    OCP_MCMD_BCST
};

// SResp encoding
enum OCPSRespType {
    OCP_SRESP_NULL = 0,
    OCP_SRESP_DVA,
    OCP_SRESP_FAIL,
    OCP_SRESP_ERR
};

// MBurstSeq encoding
enum OCPMBurstSeqType {
    OCP_MBURSTSEQ_INCR = 0,
    OCP_MBURSTSEQ_DFLT1,
    OCP_MBURSTSEQ_WRAP,
    OCP_MBURSTSEQ_DFLT2,
    OCP_MBURSTSEQ_XOR,
    OCP_MBURSTSEQ_STRM,
    OCP_MBURSTSEQ_UNKN,
    OCP_MBURSTSEQ_RESERVED
};


/////////////////////////////////////////////////////////////
// DataHSGrp holds the dataflow signals for a data hand shake
// TL1 Only
/////////////////////////////////////////////////////////////
template <class Td>
class OCPDataHSGrp
{
  public:
    typedef Td DataType;

  public:
    //
    DataType     MData;
    unsigned int MDataThreadID;
    unsigned int MDataByteEn;
    unsigned int MDataInfo;

    // OCP 2.0 Burst
    bool         MDataLast;

    // synchronization signal
    bool         MDataValid;

    // constructor
    OCPDataHSGrp(void) { reset(); }

    // define a copy constructor
    OCPDataHSGrp(const OCPDataHSGrp& src)
      : MData(src.MData),
        MDataThreadID(src.MDataThreadID),
        MDataByteEn(src.MDataByteEn),
        MDataInfo(src.MDataInfo),
        MDataLast(src.MDataLast),
        MDataValid(src.MDataValid)
    {}

    // define a copy assignment
    OCPDataHSGrp& operator=(const OCPDataHSGrp& rhs)
    {
        if (this == &rhs) { return *this; }
        copy(rhs);
        return *this;
    }

    // define a explicit copy
    void copy(const OCPDataHSGrp& src)
    {
        MData = src.MData;
        MDataThreadID = src.MDataThreadID;
        MDataByteEn = src.MDataByteEn;
        MDataInfo = src.MDataInfo;
        MDataLast = src.MDataLast;
        MDataValid = src.MDataValid;
    }

    void reset(void)
    {
        MData = 0;
        MDataThreadID = 0;
        MDataByteEn = 0xFFFFFFFF;
        MDataInfo = 0;
        MDataLast = true;
        MDataValid = false;
    }

    string print(void) { assert(0); }
};

///////////////////////////////////////////////
// RequestGrp holds the OCP request information 
///////////////////////////////////////////////
template <class Td, class Ta>
class OCPRequestGrp
{
  public:
    typedef Td DataType;
    typedef Ta AddrType;

  public:
    //
    OCPMCmdType   MCmd;
    AddrType      MAddr;
    unsigned int  MAddrSpace;
    unsigned int  MByteEn;
    unsigned int  MThreadID;
    unsigned int  MConnID;
    unsigned int  MReqInfo;

    // for OCP 2.0 Burst
    unsigned int     MAtomicLength;
    unsigned int     MBurstLength;
    bool             MBurstPrecise;
    OCPMBurstSeqType MBurstSeq;
    bool             MBurstSingleReq;
    bool             MReqLast;

    // request group with data?
    bool          HasMData;
    DataType      MData;        // TL1 Only     
    unsigned int  MDataInfo;    
    DataType      *MDataPtr;    // TL2 Only

    // constructor
    OCPRequestGrp(bool has_mdata = true) { reset(has_mdata); }

    // define a copy constructor
    OCPRequestGrp(const OCPRequestGrp& src)
      : MCmd(src.MCmd),
        MAddr(src.MAddr),
        MAddrSpace(src.MAddrSpace),
        MByteEn(src.MByteEn),
        MThreadID(src.MThreadID),
        MConnID(src.MConnID),
        MReqInfo(src.MReqInfo),
        // MBurst(src.MBurst),  // Old OCP1.0 - no longer supported
        MAtomicLength(src.MAtomicLength),
        MBurstLength(src.MBurstLength),
        MBurstPrecise(src.MBurstPrecise),
        MBurstSeq(src.MBurstSeq),
        MBurstSingleReq(src.MBurstSingleReq),
        MReqLast(src.MReqLast),
        HasMData(src.HasMData),
        MData(src.MData),
        MDataInfo(src.MDataInfo),
        MDataPtr(src.MDataPtr)
    {}

    // define a copy assignment
    OCPRequestGrp& operator= (const OCPRequestGrp& rhs)
    {
        if (this == &rhs) { return *this; }
        copy(rhs);
        return *this;
    }

    // define a explicit copy
    void copy(const OCPRequestGrp& src)
    {
        MCmd = src.MCmd;
        MAddr = src.MAddr;
        MAddrSpace = src.MAddrSpace;
        MByteEn = src.MByteEn;
        MThreadID = src.MThreadID;
        MConnID = src.MConnID;
        MReqInfo = src.MReqInfo;
        // MBurst = src.MBurst; // Old OCP 1.0 - no longer supported
        MAtomicLength = src.MAtomicLength;
        MBurstLength = src.MBurstLength;
        MBurstPrecise = src.MBurstPrecise;
        MBurstSeq = src.MBurstSeq;
        MBurstSingleReq = src.MBurstSingleReq;
        MReqLast = src.MReqLast;
        HasMData = src.HasMData;
        MData = src.MData;
        MDataInfo = src.MDataInfo;
        MDataPtr = src.MDataPtr;
    };

    void reset(bool has_mdata = true)
    {
        MCmd = OCP_MCMD_IDLE;
        MAddr = 0;
        MAddrSpace = 0;
        MByteEn = 0xFFFFFFFF;
        MThreadID = 0;
        MConnID = 0;
        MReqInfo = 0;
        // MBurst = OCP_MBURST_LAST; // Old OCP 1.0 - no longer supported
        MAtomicLength = 1;
        MBurstLength = 1;
        MBurstPrecise = true;
        MBurstSeq = OCP_MBURSTSEQ_INCR;
        MBurstSingleReq = false;
        MReqLast = true;
        HasMData = has_mdata;
        MData = 0;
        MDataInfo = 0;
        MDataPtr = NULL;
    }

    string print(void) const { assert(0); }
};

/////////////////////////////////////////////////
// ResponseGrp holds the OCP response information
/////////////////////////////////////////////////
template <class Td>
class OCPResponseGrp
{
  public:
    typedef Td DataType;

  public:
    OCPSRespType SResp;
    DataType     SData;         // TL1 Only
    DataType     *SDataPtr;     // TL2 Only
    unsigned int SDataInfo;
    unsigned int SThreadID;
    unsigned int SRespInfo;
    bool         SRespLast;

    // constructor
    OCPResponseGrp(void) { reset(); }
 
    // define a copy constructor
    OCPResponseGrp(const OCPResponseGrp& src)
      : SResp(src.SResp),
        SData(src.SData),
        SDataPtr(src.SDataPtr),
        SDataInfo(src.SDataInfo),
        SThreadID(src.SThreadID),
        SRespInfo(src.SRespInfo),
        SRespLast(src.SRespLast)
    {}
 
    // define a copy assignment
    OCPResponseGrp& operator= (const OCPResponseGrp& rhs)
    {
        if (this == &rhs) { return *this; }
        copy(rhs);
        return *this;
    }

    // define a explicit copy
    void copy(const OCPResponseGrp& src)
    {
        SResp = src.SResp;
        SData = src.SData;
        SDataPtr = src.SDataPtr;
        SDataInfo = src.SDataInfo;
        SThreadID = src.SThreadID;
        SRespInfo = src.SRespInfo;
        SRespLast = src.SRespLast;
    }

    void reset(void)
    {
        SResp = OCP_SRESP_NULL;
        SData = 0;
        SDataPtr = NULL;
        SDataInfo = 0;
        SThreadID = 0;
        SRespInfo = 0;
        SRespLast = true;
    }
    string print(void) const { assert(0); }
};

//=====================================================//
// Code Below this line used by the TL2 Channel only


/////////////////////////////////////////////////
// SideBandGrp holds the OCP sideband information
// TL2 Channel Only
/////////////////////////////////////////////////

class OCPSidebandGrp {
  public:

    // reset signals
    bool MReset_n;
    bool SReset_n;

    // system/core signals
    unsigned int  Control;
    bool ControlWr;
    bool ControlBusy;
    unsigned int  Status;
    bool StatusRd;
    bool StatusBusy;

    // sideband signals
    unsigned int  MFlag;
    bool MError;
    unsigned int  SFlag;
    bool SError;
    bool SInterrupt;


    void reset()
    {
      SInterrupt = false;
      SError = false;
      MError = false;
      MFlag = 0;
      SFlag = 0;
      Control = 0;;
      ControlWr = false;
      ControlBusy = false;
      Status = 0;
      StatusRd = false;
      StatusBusy = false;
    }

    OCPSidebandGrp()
    {
      MReset_n = true;
      SReset_n = true;
      reset();
    }
};

/////////////////////////////////////////////////
// ThreadBusyGrp holds the OCP Threadbusy information
// TL2 Channel Only
/////////////////////////////////////////////////

class OCPThreadBusyGrp {
  public:

    // Threadbusy signals
    unsigned int MThreadBusy;
    unsigned int MDataThreadBusy;
    unsigned int SThreadBusy;

    void reset()
    {
      MThreadBusy = false;
      MDataThreadBusy = false;
      SThreadBusy = false;
    }

    OCPThreadBusyGrp()
    {
      reset();
    }
};

#endif // _OCP_GLOBALS_H
