// 
//  (c) Copyright OCP-IP 2003
//  OCP-IP Confidential and Proprietary
//
// ============================================================================
//      Project : OCP SLD WG, OCP Transaction Level Layer-1
//       Author : Norman Weyrich, Synopsys Inc., weyrich@synopsys.com
//                Anssi Haverinen, Nokia Inc., anssi.haverinen@nokia.com
//                Joe Chou, Sonics Inc., joechou@sonicsinc.com 
//                Alan Kamas, for Sonics Inc., aok@sonicsinc.com, www.kamas.com
//         Date: 07/17/2003
//
//  Description : OCP Transaction Level Generic Channel
//    The Generic Channel can be used at all three Transaction Levels.
//    It is based on the generic TL channel released to OSCI
//    (www.systemc.org).
//
//    The Channel is quite generic still, and can be used to implement
//    other interface protocols with different data classes.
//
//    For information about OCP specific versions of this channel, please
//    see the files "ocp_tl1_channel.h" and "ocp_tl1_data_cl.h" and 
//    "ocp_tl2_data_cl.h".
//
//    The Generic TL Channel implements the communication between user-written
//    modules. In particular the Channel implements a 
//    Master and a Slave interface. The main features are:
//    - Two way communication: The Generic Channel consists of a
//      . Request Channel, modeling the Master to Slave transfer,
//      . Data Request Channel, modeling the Master to Slave data transfer and a
//      . Response Channel, modeling the Slave to Master transfer
//    - Blocking and non-blocking methods: Read and write methods
//      implemented in the Channel can be blocking or non-blocking.
//    - Direct access methods: Read and write methods to transport
//      data without using any resources and without affecting
//      the timing. This is useful for initialization, debugging or
//      fast access to pure functional Slave.
//    - User configurable data class: The Generic Channel gives Master
//      and Slave access to the user configurable data class.
//      This data class can be used to pass on data and
//      control information between Master and Slave.
//      The data class is a template argument allowing the usage
//      of different data classes in a system.
//    - User configurable parameter class: The parameter class
//      is intented to pass on parameters or setup values like
//      Slave addresses or Master priorities to other modules e.g. the Bus.
//    - Synchronization states and events used in the Channel
//      can be exported and hence made visible to the attached
//      modules by means of the provided communication class.
//
//   The flow of events in the Generic Channel:
//   a) Request Channel
//   - Master submits request for Slave to the Channel.
//     . Channel is blocked until Slave frees the Channel.
//     . If Master used non-blocking method the Master can probe
//       if Channel has been freed by the Slave.
//     . If Master used blocking method, the Master is blocked until the
//       Slave frees the Channel.
//   - The Generic Channel triggers Slave that there is a pending request.
//     . Slave can be made sensitive to a Channel event.
//     . Slave can probe the Channel for a pending request.
//   - Slave receives the pending request and frees the Channel
//   - After the Request Channel has been freed by the Slave the Master
//     is allowed to submit another request.
//   b) Data Request Channel
//   The Data Request Channel works the same as the Request Channel;
//   it is meant for implementing interfaces which use separate data handshake
//   c) Response-Channel
//   The Slave-Response Channel works the same as the Request Channel,
//   except that the roles of Master and Slaves are inverted. That is, the
//   Slave submits the response to the Master and the Master must free the
//   Response Channel for the next Slave response.
//     
// Parameters    :
//   Template arguments.
//     - TdataCl: Data class containing data members and access methods for
//                the data exchanged between Masters and Slaves
//     - Td: Data type
//     - Ta: Address type
//   Constructor arguments.
//     - sc_module_name: Name of the module (channel) instance
//     - Synchron: Specifies if the Channel should do a synchronous
//         update (=true) or and asynchronous update (=false)
//         of the Channel events and states.
//         Use the default value unless you know exactly
//         what you are doing.
//     - SyncEvents: Specifies if the Channel should use synchronization
//         events for the communication scheme. The Channel is faster if
//         no synchronization events are used. This could be used e.g.
//         in a Layer-1 application where only non-blocking methods
//         are employed. If blocking methods are used SyncEvents must
//         be true.
//         Use the default value (= true) unless you know exactly
//         what you are doing.
//     - DefaultEvents: Specifies if the Channel should trigger the
//         default event. The Channel is faster if no default event is
//         triggered. DefaultEvent should be set to false if the
//         attached modules are not sensitve to their ports, with other words,
//         if at least one attached module is sensitive to the port,
//         DefaultEvent should be set to true.
//         Use the default value (= true) unless you know exactly
//         what you are doing.
//                 
// =========================================================================

#ifndef _TL_CHANNEL_H
#define _TL_CHANNEL_H

#include <math.h>
#include <stdio.h>
// Changed from #include <fstream.h>
#include <fstream>
// Added for gcc 3.3:
using namespace std;

#include "systemc.h"

#include "tl_master_if.h"
#include "tl_slave_if.h"


template<class TdataCl> class TL_Channel
: public sc_module
, public TLmasterIF<TdataCl>
, public TLslaveIF<TdataCl>
{

    //---------------------------------------------------------------
    //
    // Synchronizer class. This class is used to perform a synchronous update
    // of the states and events which control the communication between
    // Master and Slave. In the asynchronouse case the update() method
    // is executed directly from the module TL_Channel, while in the synchronous
    // case this sc_prim_channel is used to call the request_update() method.
    //
    //---------------------------------------------------------------
    class Synchronizer : public sc_prim_channel
    {
        public:
            Synchronizer(TL_Channel* Channel) : m_Channel (Channel){}
            void Update() {request_update();}
        protected:
            virtual void update() {m_Channel->update();}
            TL_Channel* m_Channel;
    };


    public:

    typedef typename TdataCl::DataType Td;
    typedef typename TdataCl::AddrType Ta;

    SC_HAS_PROCESS(TL_Channel);

    //---------------------------------------------------------------
    //
    // Constructor
    //
    //---------------------------------------------------------------
    TL_Channel(sc_module_name name
            , bool Synchron     = true
            , bool SyncEvent    = true
            , bool DefaultEvent = true
            )
        :sc_module        (name)
        , m_Master         (0)
        , m_Slave          (0)
        , m_Synchron       (Synchron)
        , m_SyncEvent      (SyncEvent)
        , m_DefaultEvent   (DefaultEvent)
        , m_EventSelectFw  (0)
        , m_EventSelectFwD (0)
        , m_EventSelectBw  (0)
        , m_MdirectIF      (0)
        , m_SdirectIF      (0)
        , m_Synchronizer   (this)
        {
#ifdef TRACE_C
            outFile = new ofstream(name);
#endif

            if (!m_Synchron) {
                const char * my_name = name;
                cout << "Channel " << my_name << ": ERROR: OCP channel model currently does not support Asynchronuos mode" << endl;
                // Give up - user is expect channel to behave in a way that is not supported.
                exit(1000);
            }
            m_DataCl  = new TdataCl();
            m_CommCl  = new CommCl();
            m_ParamCl = new ParamCl<TdataCl>();

            SC_METHOD(ReqEnd);
            sensitive << m_CommCl->RequestEndEventN;
            dont_initialize();

            SC_METHOD(ResEnd);
            sensitive << m_CommCl->ResponseEndEventN;
            dont_initialize();

            SC_METHOD(DataReqEnd);
            sensitive << m_CommCl->DataRequestEndEventN;
            dont_initialize();

            // TODO: May be more efficient way to do these. Ideally, they should never
            // be triggered if the accept signal is part of the channel.
            SC_METHOD(AutoRequestAccept);
            sensitive << m_CommCl->RequestStartEvent;
            dont_initialize();

            SC_METHOD(AutoDataRequestAccept);
            sensitive << m_CommCl->DataRequestStartEvent;
            dont_initialize();

            SC_METHOD(AutoResponseAccept);
            sensitive << m_CommCl->ResponseStartEvent;
            dont_initialize();
        }

    //---------------------------------------------------------------
    //
    // Destructor
    //
    //---------------------------------------------------------------
    virtual ~TL_Channel()
    {
#ifdef TRACE_C
        delete outFile;
#endif

        delete m_DataCl;
        delete m_CommCl;
        delete m_ParamCl;
    }

    //---------------------------------------------------------------
    //
    // Reset Method
    // Resets Channel
    //
    //---------------------------------------------------------------

    virtual void reset()
    {
        // reset the channel state
        m_EventSelectFw = 0;
        m_EventSelectFwD = 0;
        m_EventSelectBw = 0;

        // reset objects owned by the channel
        m_DataCl->Reset();
        m_CommCl->reset();
    }

    //---------------------------------------------------------------
    //
    // SC_METHOD/SC_THREAD methods.
    // Processes implementing a delayed Channel release.
    //
    //---------------------------------------------------------------
    void ReqEnd()   { RequestUpdateFw(2); }
    void ResEnd()   { RequestUpdateBw(2); }
    void DataReqEnd()   { RequestUpdateFwD(2); }

    //---------------------------------------------------------------
    //
    // Register ports and perform static check.
    //
    //---------------------------------------------------------------
    virtual void register_port(sc_port_base& port, const char* if_typename)
    {
        sc_string nm(if_typename);
        if (nm == typeid(TLmasterIF<TdataCl> ).name())
        {
            // only one master can be connected
            assert(m_Master == 0);
            m_Master = &port;
        }
        else if (nm == typeid(TLslaveIF<TdataCl> ).name())
        {
            // only one slave can be connected
            assert(m_Slave == 0);
            m_Slave = &port;
        }
    }

    //---------------------------------------------------------------
    //
    // Externally visible events.
    // Event is triggered at Sput and Mput.
    // It allows making Master/Slave sensitive to the Master/Slave port(s).
    //
    //---------------------------------------------------------------
    const sc_event& default_event() const { 
        return (m_CommCl->StartEvent); 
    }

    //---------------------------------------------------------------
    //
    // Data interface access methods.
    // These methods return a pointer to the corresponding class.
    // This gives Master/Slaves access to the data structures.
    // The data structures must be provided by the user.
    //
    //---------------------------------------------------------------
    TdataCl          *GetDataCl()  { return m_DataCl; }
    ParamCl<TdataCl> *GetParamCl() { return m_ParamCl; }

    //---------------------------------------------------------------
    // Returns pointer to communication class, which
    // contains state variables and channel events.
    // These maybe used to create richer transactions by the user
    //---------------------------------------------------------------
    CommCl           *GetCommCl()  { return m_CommCl; }


    //---------------------------------------------------------------
    //
    // Explicit Channel release methods.
    // These methods unlock the Channel.
    // Srelease() unlocks the first request channel.
    // Mrelease() unlocks the first response channel.
    // SreleaseData() unlocks the second request channel (for data handshake)
    //
    //---------------------------------------------------------------
    void Mrelease(sc_time time) {m_CommCl->ResponseEndEventN.notify(time);}
    void Srelease(sc_time time) {m_CommCl->RequestEndEventN.notify(time);}
    void Mrelease(void) { RequestUpdateBw(2); }
    void Srelease(void) { RequestUpdateFw(2); }
    void SreleaseData(sc_time time) {m_CommCl->DataRequestEndEventN.notify(time);}
    void SreleaseData(void) { RequestUpdateFwD(2); }

    //---------------------------------------------------------------
    //
    // Preemptive Channel release methods.
    // These methods unlock the channel immediately after the
    // request/response has been posted
    // SreleasePE() unlocks the first request channel.
    // MreleasePE() unlocks the first response channel.
    // SreleaseDataPE() unlocks the second request channel.
    // 
    // These are meant to be used in TL1 with fully synchronous
    // masters and slaves, which want to call release before they
    // receive request.
    //---------------------------------------------------------------
    void MreleasePE(void) {wait(SC_ZERO_TIME); MgetResponseBlocking(true);}
    void SreleasePE(void) {wait(SC_ZERO_TIME); SgetRequestBlocking(true);} 
    void SreleaseDataPE(void) {wait(SC_ZERO_TIME); SgetDataRequestBlocking(true);}

    //---------------------------------------------------------------
    //
    // Methods for the master to check if the channel is busy.
    //
    //---------------------------------------------------------------
    bool MgetSbusy() const
    {
        // Is the channel occupied with a request?
        // The channel is busy if there is a request in progress, or a request pending
        return (m_CommCl->RequestStart || m_CommCl->RequestPending || m_CommCl->BlockingRequestPending);
    }

    bool MgetSbusyData() const
    {
        // Is the channel occupied with a data handshake?
        // The channel is busy if there is a data handshake in progress or pending.
        return (m_CommCl->DataRequestStart || m_CommCl->DataRequestPending || m_CommCl->BlockingDataRequestPending);
    }

    bool SgetMbusy() const
    {
        // Is the channel occupied with a response?
        // The channel is busy if there is a response in progress or pending.
        return (m_CommCl->ResponseStart || m_CommCl->ResponsePending || m_CommCl->BlockingResponsePending); 
    }

    //---------------------------------------------------------------
    //
    // Method for Slave to check if the request is a read or write
    // request. Note that the user must implement the underlying 
    // m_DataCl->IsWriteRequest() method in the data interface class.
    //
    //---------------------------------------------------------------
    bool IsWrite() { return m_DataCl->IsWriteRequest(); }


    //---------------------------------------------------------------
    //
    // Direct interface methods (for debugging).
    // Note that the MputDirect method must be implemented in the
    // Slave while the SputDirect method (if used) must be implemented
    // in the Master.
    //
    //---------------------------------------------------------------
    virtual bool MputDirect(
            int MasterID, bool IsWrite, Td *Data, Ta Address, int NumWords)
    {
        if (m_SdirectIF != 0)
            return(m_SdirectIF->MputDirect(MasterID, IsWrite, Data, Address, NumWords));
        else
            return(false);
    }

    virtual bool SputDirect(
            int SlaveID, bool IsWrite, Td *Data, Ta Address, int NumWords)
    {
        if (m_MdirectIF != 0)
            return(m_MdirectIF->SputDirect(SlaveID, IsWrite, Data, Address, NumWords));
        else
            return(false);
    }

    void SregisterDirectIF(SdirectIF<TdataCl> *A) { m_SdirectIF = A; }
    void MregisterDirectIF(MdirectIF<TdataCl> *A) { m_MdirectIF = A; }


    //---------------------------------------------------------------
    //
    // Methods for the Master to initiate a write or read transfer.
    // There exists non-blocking and blocking versions.
    //
    //---------------------------------------------------------------
    bool MputWriteRequest() 
    {
        bool tmp;
        if ((tmp = !MgetSbusy())) {
            m_DataCl->SetWriteRequest();
            tmp = MputRequest();
        }
        return tmp;
    }
    bool MputReadRequest() 
    {
        bool tmp;
        if ((tmp = !MgetSbusy())) {
            m_DataCl->SetReadRequest();
            tmp = MputRequest();
        }
        return tmp;
    }
    bool MputWriteRequestBlocking() 
    {
        m_DataCl->SetWriteRequest(); return MputRequestBlocking();
    }

    bool MputReadRequestBlocking() 
    {
        m_DataCl->SetReadRequest(); return MputRequestBlocking();
    } 


    //---------------------------------------------------------------
    //
    // The following 4 methods implement the complete communication scheme.
    // For each of the 4 methods there exist a blocking and non-blocking version.
    //
    // The above Mput methods are wrappers around the MputRequest() methods.
    //
    // It may be useful to call the MputRequest() method directly in order
    // to avoid the calling of SetWriteRequest() or SetReadRequest().
    //
    // MputRequest():  Master method to initiate a request (read or write)
    // SgetRequest():  Slave method to read a request
    // SputResponse(): Slave method to initiate a reponse
    // MgetResponse(): Master method to read the response
    //
    //---------------------------------------------------------------
    // Master method to initiate a request (read or write)
    bool MputRequest()
    {
        // First, is there already a request in progress or about to start?
        if (MgetSbusy()) {
            // A request is in progress, about to be in progress, or waiting to
            // be about to be in progress. Cannot send request now.
            return false;
        } 

        // The channel is free.
        // NOTE: it is left to the user to have checked for thread busy

        // Lock the channel from any other new requests
        m_CommCl->RequestPending = true;

        // This should already have been done.
        // Wait until the request is actually on the channel (update) before clearing this.
        // m_CommCl->RequestEnd     = false;

        // for older pre-emptive release commands
        m_CommCl->RequestPendingPE[m_CommCl->RequestTogglePE] = true; // For TL1 pre-emptive release

        // the new request will be put on the channel at the end of this delta-cycle
        RequestUpdateFw(1); // _RequestStartEvent

        // The release should be done at the delta cycle after the request is placed on the channel.
        // The code below makes the request and the release simultaneous
        // if (!m_ParamCl->CmdAccept) SreleasePE();

        return(true);
    }

    bool MputRequestBlocking()
    {
        if (m_CommCl->BlockingRequestPending) 
        {
            // Blocking request already started - abort.
            return false;
        }

        // Mark that there is a Blocking Request in progress (us)
        m_CommCl->BlockingRequestPending = true;


        // Is there a request in progress right now?
        if (m_CommCl->RequestPending || m_CommCl->RequestStart) 
        {
            // There is a request in progress
            // We must wait until it has finished
// #ifndef NDEBUG
// cout << name() << ": Master waits put request" << endl;
// #endif
            wait(m_CommCl->RequestEndEvent);

            // At this point the previous command has been accepted.
            // Note that the generic channel has no notion of time,
            // Thus we can issue multiple commands in the same cycle.
        }

        // Blocking is over. We can make our request now.
        m_CommCl->BlockingRequestPending = false;
        bool myResult = MputRequest();

        if (myResult == false) {
            // Give up - could not start request
            return false;
        }

        // Now that the command has been placed on the channel, wait around until it is accepted as well.
// #ifndef NDEBUG
//         cout << name() << ": Master waits put request" << endl;
// #endif
        wait(m_CommCl->RequestEndEvent);
// #ifndef NDEBUG
//         cout << name() << ": Master finished waiting put request" << endl;
// #endif
        return myResult;
    }


    //---------------------------------------------------------------
    //
    // Methods for the Slave to get a Master request.
    // There exist non-blocking and blocking versions.
    //
    //---------------------------------------------------------------
    bool SgetRequest(bool Release)
    {
        // Is there a current request that can be read?
        if ( !(m_CommCl->RequestStart)) {
            // There is no request to be read
            return false;
        }

        // Is there a blocking get request in progress? 
        // If so, that getRequest command has priority.
        if ( m_CommCl->BlockingGetRequestPending ) {
            return false;
        }

        // Are we the first to read this request?
        if ( !(m_CommCl->RequestUnread) ) {
            // This request has been previously read
            // return false to show that there was a problem.
            return false;
        }

        // All tests passed. This request belongs to us.

        // We have read this request. Reset the helper flag.
        m_CommCl->RequestUnread = false;

        // If the user asked for it, accept the request we have just read.
        if (Release)
        {
            Srelease();
        }

        return true;
    }

    bool SgetRequestBlocking(bool Release)
    {
        // SgetRequestBlocking rewritten to use new channel state
        // variables and to reuse SgetRequest function.

        // Is there a blocking get request already in progress? 
        // If so, that getRequestBlocking command has priority to read first.
        if ( m_CommCl->BlockingGetRequestPending ) {
            return false;
        }

        // We are the blocking getRequest. Mark it so that no one else can go.
        m_CommCl->BlockingGetRequestPending = true;

        // Is there a request for us to read?
        if ( !(m_CommCl->RequestStart) || !(m_CommCl->RequestUnread) ) {
            // Either there is not request for us (requestStart = false)
            // or someone has already read it (RequestUnread = false)
            // We will have to wait for the next request. Must be patient.
// #ifndef NDEBUG
//             cout << name() << ": Slave waits get request" << endl;
// #endif
            wait(m_CommCl->RequestStartEvent);
// #ifndef NDEBUG
//             cout << name() << ": Slave finished waiting get request" << endl;
// #endif
        }

        // There should be a request now
        assert( m_CommCl->RequestStart == true );
        assert( m_CommCl->RequestUnread == true );

        // Blocking is over - get the request normally
        m_CommCl->BlockingGetRequestPending = false;
        return SgetRequest(Release);
    }

    // NOTE: This command was not updated to new channel model
    bool SgetRequestPE() // For use with SreleasePE(). Do not mix with SgetRequest()
    {
        if (m_CommCl->RequestPendingPE[1-m_CommCl->RequestTogglePE])
        {
            // update synchronization
            m_CommCl->RequestPendingPE[1-m_CommCl->RequestTogglePE] = false;
            return(true);
        }
        else return(false);
    }

    //---------------------------------------------------------------
    //
    // Methods for the Slave to answer to a Master request.
    // There exist non-blocking and blocking versions.
    //
    //---------------------------------------------------------------

    // Slave method to initiate a reponse
    bool SputResponse()
    {
        // First, is there already a response in progress or about to start?
        if (SgetMbusy()) {
            // A response is in progress, about to be in progress, or waiting to
            // be about to be in progress. Cannot send response now.
            return false;
        } 

        // The channel is free.
        // NOTE: it is left to the user to have checked for thread busy
        // NOTE: May need to add check here in case of threadbusy_exact

        // Lock the channel from any other new responses
        m_CommCl->ResponsePending = true;

        // NOTE: for determinism, wait until the update to reset this flag
        // This should already have been done.
        // m_CommCl->ResponseEnd     = false;

        // NOTE: this is outdated - remove when PE commands go.
        // for older pre-emptive release commands
        m_CommCl->ResponsePendingPE[m_CommCl->ResponseTogglePE] = true; // For TL1 pre-emptive release

        // the new response will be put on the channel at the end of this delta-cycle
        RequestUpdateBw(1); // _ResponseStartEvent

        return(true);
    }

    bool SputResponseBlocking()
    {
        if (m_CommCl->BlockingResponsePending) 
        {
            // Blocking response already started - abort.
            return false;
        }

        // Mark that there is a Blocking Response in progress (us)
        m_CommCl->BlockingResponsePending = true;


        // Is there a response in progress right now?
        if (m_CommCl->ResponsePending || m_CommCl->ResponseStart) 
        {
            // There is a response in progress
            // We must wait until it has finished
//#ifndef NDEBUG
//            cout << name() << ": Master waits put response" << endl;
//#endif
            wait(m_CommCl->ResponseEndEvent);

            // At this point the previous response has been accepted.
            // Note that the generic channel has no notion of time,
            // Thus we can issue multiple responses in the same cycle.
        }

        // Blocking is over. We can send our response now.
        m_CommCl->BlockingResponsePending = false;
        bool myResult = SputResponse();

        if (myResult == false) {
            // give up.
            return false;
        }

        // Now that the response has been placed on the channel, wait around until it is accepted as well.
// #ifndef NDEBUG
//         cout << name() << ": Slave waits put response" << endl;
// #endif
        wait(m_CommCl->ResponseEndEvent);
// #ifndef NDEBUG
//         cout << name() << ": Slave finished waiting put response" << endl;
// #endif
        return myResult;
    }

    //---------------------------------------------------------------
    //
    // Methods for the Master to request the response from the Slave.
    // There exist non-blocking and blocking versions.
    //
    //---------------------------------------------------------------
    bool MgetResponse(bool Release)
    {
        // Is there a current response that can be read?
        if ( !(m_CommCl->ResponseStart)) {
            // There is no response to be read
            return false;
        }

        // Is there a blocking get response in progress? 
        // If so, that getResponse command has priority.
        if ( m_CommCl->BlockingGetResponsePending ) {
            return false;
        }

        // Are we the first to read this response?
        if ( !(m_CommCl->ResponseUnread) ) {
            // This response has been previously read
            // return false to show that there was a problem.
            return false;
        }

        // All tests passed. This response belongs to us.

        // We have read this response. Reset the helper flag.
        m_CommCl->ResponseUnread = false;

        // If the user asked for it, accept the response we have just read.
        if (Release)
        {
            Mrelease();
        }

        return true;
    }

    bool MgetResponseBlocking(bool Release)
    {
        // SgetResponeBlocking rewritten to use new channel state
        // variables and to reuse MgetResponse function.

        // Is there a blocking get response already in progress? 
        // If so, that getResponseBlocking command has priority to read first.
        if ( m_CommCl->BlockingGetResponsePending ) {
            return false;
        }

        // We are the blocking getResponse. Mark it so that no one else can go.
        m_CommCl->BlockingGetResponsePending = true;

        // Is there a response for us to read?
        if ( !(m_CommCl->ResponseStart) || !(m_CommCl->ResponseUnread) ) {
            // Either there is not resposne for us (requestStart = false)
            // or someone has already read it (ReponseUnread = false)
            // We will have to wait for the next response. Must be patient.
// #ifndef NDEBUG
//             cout << name() << ": Master waits get response" << endl;
// #endif
            wait(m_CommCl->ResponseStartEvent);
// #ifndef NDEBUG
//             cout << name() << ": Master finished waiting get response" << endl;
// #endif
        }

        // There should be a response now
        assert( m_CommCl->ResponseStart == true );
        assert( m_CommCl->ResponseUnread == true );

        // Blocking is over - get the response normally
        m_CommCl->BlockingGetResponsePending = false;
        return MgetResponse(Release);
    }

    // NOTE: This function is outdated and not needed for the new channel model
    bool MgetResponsePE()// For use with MreleasePE(). Do not mix with MgetResponse()
    {
        if (m_CommCl->ResponsePendingPE[1-m_CommCl->ResponseTogglePE])
        {
            // update synchronization
            m_CommCl->ResponsePendingPE[1-m_CommCl->ResponseTogglePE] = false;
            return(true);
        }
        else return(false);
    }

    //---------------------------------------------------------------
    //
    // Methods for the master to post a request on
    // second channel, used for data handshake
    // There exist non-blocking and blocking versions.
    //
    //---------------------------------------------------------------
    bool MputDataRequest()
    {
        // First, is there already a data handshake in progress or about to start?
        if (MgetSbusyData()) 
        {
            // A data handshake is in progress, about to be in progress, 
            // or waiting to be about to be in progress. 
            // Cannot send request now.
            return false;
        } 

        // The data channel is free.
        // NOTE: it is left to the user to have checked for data thread busy

        // Lock the channel from any other new data requests
        m_CommCl->DataRequestPending = true;

        // To keep the system deterministic, wait for update to reset this signal
        // This should already have been done.
        // m_CommCl->DataRequestEnd     = false;

        // for older pre-emptive release commands
        m_CommCl->DataRequestPendingPE[m_CommCl->DataRequestTogglePE] = true; // For TL1 pre-emptive release

        // the new request will be put on the channel at the end of this delta-cycle
        RequestUpdateFwD(1); // DataStartEvent
        return(true);
    }

    bool MputDataRequestBlocking()
    {
        if (m_CommCl->BlockingDataRequestPending) 
        {
            // Blocking data already started - abort.
            return false;
        }

        // Mark that there is a Blocking Request in progress (us)
        m_CommCl->BlockingDataRequestPending = true;


        // Is there a data request in progress right now?
        if (m_CommCl->DataRequestPending || m_CommCl->DataRequestStart) 
        {
            // There is a data request in progress
            // We must wait until it has finished
// #ifndef NDEBUG
//             cout << name() << ": Master waits put data" << endl;
// #endif
            wait(m_CommCl->DataRequestEndEvent);

            // At this point the previous data has been accepted.
            // We now need to wait for the next clock cycle to start a new data request.
            wait();
        }

        // NOTE: no check for thread busy here. This may not be the expected
        //       behavior.

        // Blocking is over. We can make our request now.
        m_CommCl->BlockingDataRequestPending = false;
        bool myResult = MputDataRequest();

        if (myResult == false) {
            // give up.
            return false;
        }

        // Now that the data has been placed on the channel, wait around until it is accepted as well.
// #ifndef NDEBUG
//         cout << name() << ": Master waits put data request" << endl;
// #endif
        wait(m_CommCl->DataRequestEndEvent);
// #ifndef NDEBUG
        cout << name() << ": Master finished waiting put data request" << endl;
// #endif
        return myResult;
    }

    //---------------------------------------------------------------
    //
    // Methods for the slave to get a master request,
    // second channel, used for data handshake
    // There exist non-blocking and blocking versions.
    //
    //---------------------------------------------------------------
    bool SgetDataRequest(bool Release)
    {
        // Is there current data that can be read?
        if ( !(m_CommCl->DataRequestStart)) {
            // There is no data to be read
            return false;
        }

        // Is there a blocking get data in progress? 
        // If so, that command has priority.
        if ( m_CommCl->BlockingGetDataRequestPending ) {
            return false;
        }

        // Are we the first to read this data?
        if ( !(m_CommCl->DataRequestUnread) ) {
            // This data has been previously read
            // return false to show that there was a problem.
            return false;
        }

        // All tests passed. This data is ours - now get it.

        // We have read this data. Reset the helper flag.
        m_CommCl->DataRequestUnread = false;

        // If the user asked for it, accept the data we have just read.
        if (Release)
        {
            SreleaseData();
        }

        return true;
    }

    bool SgetDataRequestBlocking(bool Release)
    {
        // SgetDataRequestBlocking rewritten to use new channel state
        // variables and to reuse SgetDataRequest function.

        // Is there a blocking get data already in progress? 
        // If so, that getDataRequestBlocking command has priority to read first.
        if ( m_CommCl->BlockingGetDataRequestPending ) {
            return false;
        }

        // We are the blocking getDataRequest. Mark it so that no one else can go.
        m_CommCl->BlockingGetDataRequestPending = true;

        // Is there already data for us to read?
        if ( !(m_CommCl->DataRequestStart) || !(m_CommCl->DataRequestUnread) ) {
            // Either there is not data for us (DataRequestStart = false)
            // or someone has already read it (DataRequestUnread = false)
            // We will have to wait for the next data. Must be patient.
// #ifndef NDEBUG
//             cout << name() << ": Slave waits get data" << endl;
// #endif
            wait(m_CommCl->DataRequestStartEvent);
// #ifndef NDEBUG
//             cout << name() << ": Slave finished waiting get data" << endl;
// #endif
        }

        // There should be data now
        assert( m_CommCl->DataRequestStart == true );
        assert( m_CommCl->DataRequestUnread == true );

        // Blocking is over - get the data normally
        m_CommCl->BlockingGetDataRequestPending = false;
        return SgetDataRequest(Release);
    }

    // NOTE: this outdated command is not needed with the new channel
    bool SgetDataRequestPE() // For use with SreleaseDataPE(). Do not mix with SgetDataRequest()
    {
        if (m_CommCl->DataRequestPendingPE[1-m_CommCl->DataRequestTogglePE])
        {
            // update synchronization
            m_CommCl->DataRequestPendingPE[1-m_CommCl->DataRequestTogglePE] = false;
            return(true);
        }
        else return(false);
    }

    //---------------------------------------------------------------
    // Update method. This method is used to update the states and events
    // which control the communication between Master and Slave.
    // This method is not part of the API. However it is public because
    // it is used by the Synchronizer class.
    // update() use delayed notify 
    //---------------------------------------------------------------

    // ---------------------------------------------------------
    // update()
    // Performs an end of delta cycle update for each of the channel phases
    // Request = m_EventSelectFw
    // Data handshake = m_EventSelectFwD
    // Response = m_EventSelectBw
    // ---------------------------------------------------------
    void update()
    {
#ifdef TRACE_C
        double ttime;
#endif

        switch(m_EventSelectFw)
        {
            case 1: 
                // Start a request to the channel
                if (m_SyncEvent) m_CommCl->RequestStartEvent.notify(SC_ZERO_TIME);
                m_CommCl->RequestPending = false;
                m_CommCl->RequestStart = true;
                m_CommCl->RequestUnread = true;
                // Make sure that RequestEnd is false - we are starting a new Request phase.
                m_CommCl->RequestEnd = false;
                if (m_DefaultEvent) m_CommCl->StartEvent.notify(SC_ZERO_TIME);

                // Support for PE Commands
                m_CommCl->RequestTogglePE = 1 - m_CommCl->RequestTogglePE;

                // now update the data class to put the new request on the channel
                m_DataCl->update_Fw(1);

#ifdef TRACE_C
                ttime = sc_time_stamp().to_seconds();
                *outFile<<" "<<m_DataCl->SgetMThreadID()<<" Req start at "<<ttime<<endl;
#endif

                break;
            case 2:
                // Accept the request and release the channel
                // NOTE: this logic might be incorrect. The Request is over, so how
                //       can RequestEnd be false?? We just put out the RequestEnd Event
                //       so the corresponding signal should match.
                // if (m_CommCl->RequestPending == false)
                // m_CommCl->RequestEnd = true;
                // else
                //   m_CommCl->RequestPending = false;
                m_CommCl->RequestStart = false;
                m_CommCl->RequestEnd = true;
                m_CommCl->RequestEndTime = sc_simulation_time();
                if (m_SyncEvent) m_CommCl->RequestEndEvent.notify(SC_ZERO_TIME);
                // Check for requests that were never read
                if (m_CommCl->RequestUnread) {
#ifndef NDEBUG
                    cout << name() << ": WARNING - request accepted but never read." << endl;
#endif
                    m_CommCl->RequestUnread = false;
                }
#ifdef TRACE_C
                ttime = sc_time_stamp().to_seconds();
                *outFile<<" "<<m_DataCl->SgetMThreadID()<<" Req end at "<<ttime<<endl;
#endif
                // Call the data channel to make any changes needed to release
                // the request.
                m_DataCl->update_Fw(2);
                break;
            default: 
                break;
        }
        // updates for the datahandshake phase
        switch(m_EventSelectFwD)
        {
            case 1: 
                // Start data on the channel
                if (m_SyncEvent) m_CommCl->DataRequestStartEvent.notify(SC_ZERO_TIME);
                m_CommCl->DataRequestPending = false;
                m_CommCl->DataRequestStart = true;
                m_CommCl->DataRequestUnread = true;
                // Reset the DataRequestEnd flag as we are starting a new data phase.
                m_CommCl->DataRequestEnd = false;
                if (m_DefaultEvent) m_CommCl->StartEvent.notify(SC_ZERO_TIME);

                // Support for older PE Commands
                m_CommCl->RequestTogglePE = 1 - m_CommCl->RequestTogglePE;

                // Update the data class to put the new data on the channel
                m_DataCl->update_FwD(1);

#ifdef TRACE_C
                ttime = sc_time_stamp().to_seconds();
                *outFile<<" Data Handshake started at "<<ttime<<endl;
#endif

                break;

            case 2:
                // Accept the data and release the data handshake channel
                m_CommCl->DataRequestStart = false;
                m_CommCl->DataRequestEnd = true;
                m_CommCl->DataRequestEndTime = sc_simulation_time();
                if (m_SyncEvent) m_CommCl->DataRequestEndEvent.notify(SC_ZERO_TIME);
                // Check for data that was never read
                if (m_CommCl->DataRequestUnread) {
#ifndef NDEBUG
                    cout << name() << ": WARNING - data accepted but never read." << endl;
#endif
                    m_CommCl->RequestUnread = false;
                }
#ifdef TRACE_C
                ttime = sc_time_stamp().to_seconds();
                *outFile<<" Data Handshake end at "<<ttime<<endl;
#endif

                // Update the data class now that the DataHS has been accepted
                m_DataCl->update_FwD(2);
                break;

            default: 
                break;
        }
        // updates for the Response phase
        switch(m_EventSelectBw)
        {
            case 1:
                // start a response on the channel
                if (m_SyncEvent) {
                    m_CommCl->ResponseStartEvent.notify(SC_ZERO_TIME);
                }
                m_CommCl->ResponsePending = false;
                m_CommCl->ResponseStart = true;
                m_CommCl->ResponseUnread = true;
                // Reset the ResponseEnd signal as we are now starting a new Response Phase.
                m_CommCl->ResponseEnd = false;
                if (m_DefaultEvent) m_CommCl->StartEvent.notify(SC_ZERO_TIME);

                // Support for outdated PE Commands
                m_CommCl->ResponseTogglePE = 1 - m_CommCl->ResponseTogglePE;

                // update the data class to tell it to add the new response to channel.
                m_DataCl->update_Bw(1);

#ifdef TRACE_C
                ttime = sc_time_stamp().to_seconds();
                *outFile<<"                                         "
                    <<m_DataCl->MgetSThreadID()<<" Response Start at "<<ttime<<endl;
#endif

                break;

            case 2:
                // Accept the response and release the channel
                m_CommCl->ResponseStart = false;
                m_CommCl->ResponseEnd = true;
                m_CommCl->ResponseEndTime = sc_simulation_time();
                if (m_SyncEvent) m_CommCl->ResponseEndEvent.notify(SC_ZERO_TIME);
                // Check for responses that were never read
                if (m_CommCl->ResponseUnread) {
#ifndef NDEBUG
                    cout << name() << ": WARNING - response accepted but never read." << endl;
#endif
                    m_CommCl->ResponseUnread = false;
                }
#ifdef TRACE_C
                ttime = sc_time_stamp().to_seconds();
                *outFile<<"                                         "
                    <<m_DataCl->MgetSThreadID()<<" Response end at "<<ttime<<endl;
#endif

                // Update the data class now that the reponse has been accepted 
                m_DataCl->update_Bw(2);
                break;

            default: 
                break;
        }
        // Reset dataflow update state and set up any follow-up events
        // (such as automatic accepts) for the next delta cycle.
        m_EventSelectFwD = 0;
        m_EventSelectFw = 0;
        m_EventSelectBw = 0;
    }

    //------------------ End of public methods ----------------------


                protected:

    // NOTE: m_EventSelectXX should be zero when these functions are
    //       called. If now, we've got a problem of the request/response/data
    //       be put on channel and accepted at the same exact time. That is,
    //       Start and End are simultaneous. This generally will not work.
    // NOTE: could add an assert here.
    void RequestUpdateFw(int i)
    {
        m_EventSelectFw = i;
        //if (m_Synchron) m_Synchronizer.Update();
        // else AsyncUpdate();
        m_Synchronizer.Update();
    }

    void RequestUpdateFwD(int i)
    {
        m_EventSelectFwD = i;
        //if (m_Synchron) m_Synchronizer.Update();
        //else AsyncUpdate();
        m_Synchronizer.Update();
    }

    void RequestUpdateBw(int i)
    {
        m_EventSelectBw = i;
        // if (m_Synchron) m_Synchronizer.Update();
        // else AsyncUpdate();
        m_Synchronizer.Update();
    }

    // TODO: for efficiency this method should be detriggered if SRespAccept is part of the channel
    void AutoRequestAccept()
    {
        // sensitive to RequestStartEvent
        if (! m_ParamCl->cmdaccept )
        {
            Srelease();
        }    
    }
    // TODO: for efficiency this method should be detriggered if SRespAccept is part of the channel
    void AutoDataRequestAccept()
    {
        // sensitive to ResponseStartEvent
        if (! m_ParamCl->dataaccept )
        {
            SreleaseData();
        }    
    }
    // TODO: for efficiency this method should be detriggered if SRespAccept is part of the channel
    void AutoResponseAccept()
    {
        // sensitive to ResponseStartEvent
        if (! m_ParamCl->respaccept )
        {
            Mrelease();
        }    
    }

    //---------------- End of Methods --------------------------------------


    //---------------- Protected Data ----------------------------------------

    // master and slave ports to be bounded
    sc_port_base* m_Master;
    sc_port_base* m_Slave;

    // contructor parameter
    bool m_Synchron;
    bool m_SyncEvent;
    bool m_DefaultEvent;

    // Switches for controlling the update() method
    int m_EventSelectFw;   // Request
    int m_EventSelectFwD;  // Request Data
    int m_EventSelectBw;   // Response

    // Direct interface pointer
    MdirectIF<TdataCl> *m_MdirectIF;
    SdirectIF<TdataCl> *m_SdirectIF;

    // Channel to track request_update synchronization.
    Synchronizer m_Synchronizer;

    // Pointers to the data structure classes provided by the user
    TdataCl          *m_DataCl;
    ParamCl<TdataCl> *m_ParamCl;

    // Pointer to the communication class
    CommCl           *m_CommCl;

#ifdef TRACE_C
    // Output file to trace the communication
    ofstream *outFile;
#endif
}; // end module TL_Channel

#endif  // _TL_CHANNEL_H

