// Copyright (c) 2004 Sonics, Inc.
//
// Confidential and Proprietary Information of Sonics, Inc.
// Use, disclosure, or reproduction is prohibited without
// written permission from Sonics, Inc.
//
// $Id: BigInt.h,v 1.2 2007/02/05 14:22:46 guenzel Exp $
//
// Class to read a text file in lines, with chunk accesses
// without keeping file descriptor open

#ifndef _BigInt_h_
#define _BigInt_h_

#include <algorithm>
#include <string>
#include <bitset>
#include "IntTypes.h"
#include <stdexcept>
#include <sstream>

#if __GNUC__ >= 3
#define BITSET_SHIFT_WORKING
#endif

using namespace std;
namespace Sonics {
template<int N>
class BigInt : public bitset<N>
{
    typedef uint32_t Word;

    enum {
        Wordsize = sizeof( Word ),
        Charsperword = 2 * sizeof( Word )
    };
    
  public:
    BigInt( Word i=0 ) :
        bitset<N>( i )
        {}
    BigInt( const string& s ) {
        fromStr( s.begin(), s.end() );
    }
    BigInt( const char* str, const char* end ) {
        fromStr( str, end );
    }
    // operator= with size difference
    template<int M>
    BigInt& operator=( const BigInt<M>& other ) {
	bitset<N>::reset();
	int min = ( M < N ) ? M : N;
	for ( int i=0; i < min; ++i )
	    if  ( other.test( i ) ) bitset<N>::operator[](i) = 1;
        return *this;
    }
    // pass through operators
    template<typename otherT>
    bool operator==( const otherT& other ) const {
        return bitset<N>::operator==( other );
    }
    template<typename otherT>
    BigInt& operator=( const otherT& other ) {
        bitset<N>::operator=( other );
        return *this;
    }
    template<typename otherT>
    BigInt& operator&=( const otherT& other ) {
        bitset<N>::operator&=( other );
        return *this;
    }
    template<typename otherT>
    BigInt& operator|=( const otherT& other ) {
        bitset<N>::operator|=( other );
        return *this;
    }
    // shift operators to fix libgcc bug
    BigInt& operator>>=( size_t shift ) {
#ifndef BITSET_SHIFT_WORKING
        {
            BigInt<N> shifted = *this;
            unsigned int i;
            for ( i=Wordsize*4; i <= shift; i += Wordsize*4 )
                shifted.bitset<N>::operator >>= ( Wordsize*4 );
            operator=( shifted );
            if ( i > shift )
                shift = shift - ( i - Wordsize*4 );
            else
                shift = 0;
        }
#endif
        bitset<N>::operator>>=( shift );
        return *this;
    }
    BigInt& operator<<=( size_t shift ) {
#ifndef BITSET_SHIFT_WORKING
        {
            BigInt<N> shifted = *this;
            unsigned int i;
            for ( i=Wordsize*4; i <= shift; i += Wordsize*4 )
                shifted.bitset<N>::operator <<= ( Wordsize*4 );
            operator=( shifted );
            if ( i > shift )
                shift = shift - ( i - Wordsize*4 );
            else
                shift = 0;
        }
#endif
        bitset<N>::operator<<=( shift );
        return *this;
    }

    // cast operators
    operator unsigned int() const {
        BigInt<N> copy;
        copy = *this;
        if ( N > 8 * sizeof( unsigned int ) ) {
            for ( unsigned int i=8 * sizeof( unsigned int ); i<N; ++i )
                copy.reset( i );
        }
//        return copy.to_ulong();
          return copy.to_ulong();  
    }
#if !defined(__LP64__) && !defined(__arch64__)
    operator unsigned long() const {
        return ( unsigned long )( operator unsigned int()() );
    }
#endif
    operator snx_uint64_t() const {
        bitset<N> wordMask( static_cast<Word>( -1 ) );
        snx_uint64_t ret = 0;
        for ( unsigned int i=0; i < sizeof(ret)/Wordsize; ++i ) {
            unsigned int shift = Wordsize * i * 8;
#ifndef BITSET_SHIFT_WORKING
            bitset<N> partial = *this;
            for ( unsigned int j=Wordsize*4; j<=shift; j+= Wordsize*4 )
                partial >>= (Wordsize*4);
            partial &= wordMask;
#else
            bitset<N> partial = ( this->bitset<N>::operator>>(shift) ) & wordMask;
#endif
            snx_uint64_t partialLong = partial.to_ulong();   
            ret |= ( partialLong << shift );
        }
        return ret;
    }

  private:
    // construct from a hex string
    void fromStr( const char* str, const char* end ) {
        *this = 0;
        const char *scan = str;        // need to find end
        for (; scan < end ; scan++) {  // ignore trailing crap
            char tst = tolower(*scan);
            if (!isdigit(tst) && !('a' <= tst && tst <= 'f')) {
  	        if ('x' == tst && 1 == scan - str && '0' == *str) { // leading 0x OK
		    str += 2; // reposition start
                    continue;
	        }
	        // should test for proper termination here
                break;
	    }
	}
        // scan now points to the least significant nibl

        const int                  bpw   = 8 * Wordsize;
        unsigned int               words = (N + bpw - 1)/bpw;
        for ( unsigned int wordIndex=0; wordIndex < words; ++wordIndex ) {
	    Word        word = 0;
            int         bits = 0;
            for (; scan > str && bits < bpw ; bits += 4) {
	        unsigned char nibl = *--scan;
                if (isdigit(nibl))
                  nibl -= '0';
                else
		  nibl  = 10 + (tolower(nibl) - 'a');
                word |= nibl << bits;
	    }
            bitset<N> partial( word );
#ifndef BITSET_SHIFT_WORKING
            for ( unsigned int j=Wordsize*4; j<=wordIndex*bpw; j+= Wordsize*4 )
                partial <<= (Wordsize*4);
#else
            partial <<= ( wordIndex * bpw );
#endif
            (*this) |= partial;
        }
        if (scan > str) {            
            std::ostringstream err;
            err << "BigInt<" << N << "> -- value too large for type: 0x" << str << ends;
            throw std::out_of_range( err.str() );
        }
    }
};            
}

#endif /* _BigInt_h_ */
