// 
//  Copyright 2008 OCP-IP
//
// ============================================================================
//      Project : OCP SLD WG
//       Author : Albert Chitemyan Sonics Inc.
//                
//         $Id: 
//
//  Description : OCP API - Layer-2 test Master
//    Features:
//    - Non-blocking methods are used
//    - demonstration of usage sc_time argument for nb_transport()
// ============================================================================

using namespace std;
#include "master.h"

#define CLKPERIOD 1 // ns that is
#define NUM_BURST 10

// ----------------------------------------------------------------------------
// Constructor
// ----------------------------------------------------------------------------
template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH> Master<BUSWIDTH, ADDRWIDTH>::Master (sc_core::sc_module_name name_): 
  sc_core::sc_module (name_), ip_socket("ip_socket", ocpip::ocp_master_socket_tl2<32>::mm_txn_with_data()), ongoing_req(false), rsp(NULL) {
    
  scv_random::set_global_seed(scv_random::pick_random_seed());
  // initialize common members
  transaction_count = 0;

  resp_delay = sc_core::SC_ZERO_TIME;
  
  SC_THREAD(sendBurst);
  sensitive<< send_burst_ev;

  ip_socket.register_nb_transport_bw(this, &Master::nb_transport);

  SC_THREAD(receiveResp);
  sensitive<< receive_resp_ev;
  dont_initialize();

  // we will use these burst lengths in this example
  burst_len_list[0] = 4;
  burst_len_list[1] = 8;  
  burst_len_list[2] = 16;  
}

// ----------------------------------------------------------------------------
// Destructor
// ----------------------------------------------------------------------------
template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH> Master<BUSWIDTH, ADDRWIDTH>::~Master() {}

// ----------------------------------------------------------------------------
//  Method : Master::sendBurst
//
// ----------------------------------------------------------------------------
template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH>
void Master<BUSWIDTH, ADDRWIDTH>::sendBurst()
{
    while (true) 
    {
    wait(resp_delay);

    if ( transaction_count > NUM_BURST )
	return;

    // check if channel is free
    if (!ongoing_req) {

	generate_burst();

	// Send burst 
	time  = sc_core::SC_ZERO_TIME;
	phase = tlm::BEGIN_REQ;
	tlm::tlm_sync_enum slave_answer = ip_socket->nb_transport_fw(*req, phase, time);
   
	switch(slave_answer){
	case tlm::TLM_ACCEPTED: 
	    ongoing_req = true; 
	    break;
	case tlm::TLM_UPDATED: 
	    switch (phase){
	    case tlm::END_REQ:
		if (req->get_command() == tlm::TLM_WRITE_COMMAND){
		    ip_socket.release_transaction(req); //we are done now because writes do not have resps in this example
		}
		break;
	    default:
		std::cerr<<"Error in "<<name()<<" : got unexpected phase update to "<<phase<<" in response to BEGIN_REQ."<<std::endl;
		exit(1);                
	    }
	    break;
	case tlm::TLM_COMPLETED:;
	}

#ifdef DEBUG_TL2
	cout << 
	    endl << "------------------------------------------------------------------------------------ time=" 
	     << sc_core::sc_time_stamp() << endl;

	cout << "Master sent burst with length " <<  burst_len;

	if (req->get_command() == tlm::TLM_WRITE_COMMAND) {
	    cout << "   WRITE "
		 << "   addr=" << hex << req->get_address()
		 << "   first_data=" << (*((Td*)(req->get_data_ptr())));

	} else {
	    cout << "   READ "
		 << "   addr=" << hex << req->get_address();
	}

	cout << endl;
#endif
      
	transaction_count++;
    }
    
    wait();

    } // end of while
}

template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH>
void Master<BUSWIDTH, ADDRWIDTH>::receiveResp()
{
    while (true)
    {
        wait(resp_delay);
        if (rsp) 
        {
	        time = sc_core::SC_ZERO_TIME;
    	    phase = tlm::END_RESP;
    	    ip_socket->nb_transport_fw(*rsp, phase, time);  

        	if (rsp->get_response_status() == tlm::TLM_OK_RESPONSE) {
#ifdef DEBUG_TL2
	           cout << "Master got valid response "
		       << "  first_data=" << hex << (*((Td*)(rsp->get_data_ptr())))
    		   << dec << "                time=" << sc_core::sc_time_stamp() << endl;
#endif
    	    }
        	ip_socket.release_transaction(rsp); //we are done with this txn now
    	    rsp = NULL;
        }
        wait();
    }
} // end of method

template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH>
tlm::tlm_sync_enum Master<BUSWIDTH, ADDRWIDTH>::nb_transport(tlm::tlm_generic_payload& txn, tlm::tlm_phase& ph, sc_core::sc_time& tim) 
{
    scv_smart_ptr< unsigned int > rand;
    rand->next();

    switch(ph){
    case tlm::END_REQ:
	ongoing_req = false;
	send_burst_ev.notify( ( (*rand) % 3 ) * CLKPERIOD, sc_core::SC_NS );
	if (req->get_command() == tlm::TLM_WRITE_COMMAND){
	    ip_socket.release_transaction(req); //we are done now because writes do not have resps in this example
	}      
	break;
    case tlm::BEGIN_RESP: 
    resp_delay = tim;
	rsp = &txn;
    rand->next();
	receive_resp_ev.notify( ( (*rand) % 2 ) * CLKPERIOD, sc_core::SC_NS );
	break;
    default:
	std::cerr<<"Error in "<<name()<<" : got unexpected phase "<<ph<<" in nb_transport_bw"<<std::endl;
	exit(1);
    }
    return tlm::TLM_ACCEPTED;
}


template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH>
void Master<BUSWIDTH, ADDRWIDTH>::generate_burst() 
{
    scv_smart_ptr< unsigned int > rand;
    rand->next();
    req=ip_socket.get_transaction();
    tlm::tlm_command cmd;

    if ((*rand) % 2 == 0)
	cmd = tlm::TLM_WRITE_COMMAND; 
    else
	cmd = tlm::TLM_READ_COMMAND;

    rand->next();
    // burst start address
    unsigned int addr = 0x80 * ( 1 + (*rand)%5 );

    burst_len = burst_len_list[ (*rand)%3 ];

    req->set_command(cmd);
    req->set_address(addr);
    ip_socket.reserve_data_size(*req, burst_len*sizeof(Td));

    ocpip::burst_sequence* b_seq;
    ip_socket.get_extension<ocpip::burst_sequence>(b_seq, *req);
    b_seq->value.sequence = ocpip::INCR;
    ip_socket.validate_extension<ocpip::burst_sequence>(*req);

    ocpip::burst_length* b_len; 
    ip_socket.get_extension<ocpip::burst_length>(b_len, *req);
    b_len->value = burst_len;
    ip_socket.validate_extension<ocpip::burst_length>(*req);

    // generate burst_length data for entire burst
    if (req->get_command() == tlm::TLM_WRITE_COMMAND) 
    {
	unsigned int localdata = 10000 + ((*rand)/100) * 100;

	for (unsigned int i = 0; i < burst_len; ++i)
	    *(((Td*)(req->get_data_ptr())) + i) = localdata++;
    }  

}

// ----------------------------------------------------------------------------
//
//  Instantiation of the Master
//
// ----------------------------------------------------------------------------
template class Master< 32,32 >;

