// 
//  Copyright 2008 OCP-IP
//
// ============================================================================
//      Project : OCP SLD WG
//       Author : Anssi Haverinen, Nokia Inc.
//                Robert Guenzel (from TU of Braunschweig) for Greensocs Ltd.
//           $Id: slave.cc,v 1.2 2005/01/07 03:42:33 Anssi Exp $
//
//  Description : OCP API - TL1 profile example
// ============================================================================

#include "slave.h"

// ----------------------------------------------------------------------------
// Constructor
// ----------------------------------------------------------------------------
Slave::Slave (sc_core::sc_module_name name_)
  : sc_core::sc_module (name_), tpP("tpPort"), req(NULL), ongoing_resp(false), rsp(NULL) {

  // initialize common members
  address = 0;
  burststart = true;
  last_request = tlm::TLM_IGNORE_COMMAND;
  bytemask = 0;
  for (int i=0;i<1024;i++)
    memory[i] = 0;
  reqcnt=0;
  SC_METHOD(proc);
  sensitive<<clk.pos();
  dont_initialize(); // make sure we execute only at clock edge
  tpP.register_nb_transport_fw(this, &Slave::nb_transport);
  tpP.activate_delta_cycle_protection();
}


// ----------------------------------------------------------------------------
// Destructor
// ----------------------------------------------------------------------------
Slave::~Slave(){}


// ----------------------------------------------------------------------------
//  Method : Slave::proc()
//
//  Synchronous slave process
//
// ----------------------------------------------------------------------------

void Slave::proc(){

  // If a previous has been served, we check for new requests
  if (last_request == tlm::TLM_IGNORE_COMMAND) {
    if (req) {
      last_request = req->get_command();
      
      unsigned int MByteEn=0;
      byte_en_helper::setBE(MByteEn, req, sizeof(Td), reqcnt);
      
      // Byte enable to bitmask
      if (MByteEn & 0x1)
        bytemask = 0xFF; 
      if  (MByteEn & 0x2)
        bytemask = 0xFF00 | bytemask; 
      if  (MByteEn & 0x4)
        bytemask = 0xFF0000 | bytemask; 
      if  (MByteEn & 0x8)
        bytemask = 0xFF000000 | bytemask; 

      // Burst addressing
      if (burststart) {
        address = req->get_address()/4; // First of burst
        reqcnt=0;
        assert(tpP.get_extension<ocpip::imprecise>(*req));
        
        ocpip::burst_sequence* b_seq;
        if (tpP.get_extension<ocpip::burst_sequence>(b_seq, *req)) //we have a burst sequence
          current_sequence=b_seq->value.sequence;
        else
          current_sequence=ocpip::INCR;

      }
      else {
        if (current_sequence==ocpip::INCR)
          address = address+1; //incrementing address
        else //strm
        if (current_sequence==ocpip::STRM)
          address=address;      
        else
          assert(0);
      }
      reqcnt++;

      //we are allowed to check burst length in EVERY req phase, since it is an imprecise burst!
      unsigned int b_length=1;
      ocpip::burst_length* b_len;
      if (tpP.get_extension<ocpip::burst_length>(b_len, *req)) b_length=b_len->value;
      else {std::cerr<<"Imprecise burst without burst length extension! Error."<<std::endl; abort();}
      
      
      
      if (b_length>1) // can handle only imprecise bursts
        burststart = false; // burst continues
      else
        burststart = true; // end of burst, next can start

      if (last_request == tlm::TLM_READ_COMMAND) {
        rsp=req; 
#ifdef DEBUG_G1
        std::cout << "Slave got read request "
         << "  time  = " << sc_core::sc_time_stamp().to_seconds()
         << "  address  = " << rsp->get_address()<< std::endl;
#endif
      }
      else {
        rsp=req;      
#ifdef DEBUG_G1
        std::cout << "Slave got write request "
         << "  time  = " << sc_core::sc_time_stamp().to_seconds()
         << "  data  = " << (*(((Td*)(rsp->get_data_ptr()))+reqcnt-1)) << std::endl;
#endif

      }
      my_req_info* req_info;
      if (tpP.get_extension<my_req_info>(req_info, *req)){
        //the req has an info field
        std::cout<<"Slave got info for a request!"<<std::endl
                 <<"The info is:"<<std::endl
                 <<"  "<<req_info->value.infos[reqcnt-1]<<std::endl;
      }

      
      time=sc_core::SC_ZERO_TIME;
      phase=tlm::END_REQ;
      tlm::tlm_sync_enum retVal=tpP->nb_transport_bw(*req,phase, time);
      switch(retVal){
        case tlm::TLM_ACCEPTED: break; //we are only doing bursts so we don't expect any TLM2 phase magic
        case tlm::TLM_UPDATED:
          std::cerr<<"Error in "<<name()<<" got unexpected phase update "<<phase<<" in response to END_REQ"<<std::endl;
          exit(1);
          break;
        case tlm::TLM_COMPLETED:;
      }
      req=NULL;
        
    }
  }


  if (last_request == tlm::TLM_WRITE_COMMAND)
    memory[address] = (*(((Td*)(rsp->get_data_ptr()))+reqcnt-1)) & bytemask;

  if (!ongoing_resp && rsp) {
    if (last_request == tlm::TLM_READ_COMMAND)
      *(((Td*)(rsp->get_data_ptr()))+reqcnt-1) = memory[address];
    // Set OCP response
    rsp->set_response_status(tlm::TLM_OK_RESPONSE);
    
    // Send response
    time=sc_core::SC_ZERO_TIME;
    phase=tlm::BEGIN_RESP;
    tlm::tlm_sync_enum retVal=tpP->nb_transport_bw(*rsp,phase, time);
    switch(retVal){
      case tlm::TLM_ACCEPTED: ongoing_resp=true; break;
      case tlm::TLM_UPDATED:
        if (phase!=tlm::END_RESP) {
          std::cerr<<"Error in "<<name()<<" got unexpected phase update "<<phase<<" in response to BEGIN_RESP"<<std::endl;
          exit(1);
        }
        break;
      case tlm::TLM_COMPLETED: break; //we don't check that it was really the last resp. we just trust the master...
    }

#ifdef DEBUG_G1
    std::cout << "Slave sent response " << rsp->get_response_status()
     << " time " << sc_core::sc_time_stamp().to_seconds();
    if (last_request == tlm::TLM_READ_COMMAND)
      std::cout<< " data " << (*(((Td*)(rsp->get_data_ptr()))+reqcnt-1));
     std::cout<< std::endl;
#endif
    
    last_request = tlm::TLM_IGNORE_COMMAND;
    rsp=NULL;
  }
} // end of method

tlm::tlm_sync_enum Slave::nb_transport(tlm::tlm_generic_payload& txn, tlm::tlm_phase& ph, sc_core::sc_time& tim){
  switch(ph){
    case tlm::BEGIN_REQ: req=&txn; break;
    case tlm::END_RESP: ongoing_resp=false; break;
    default:
      std::cerr<<"Error in "<<name()<<" got unexpected phase "<<phase<<" in nb_transport_fw"<<std::endl;
      exit(1);        
  }
  return tlm::TLM_ACCEPTED;
}


