//
//  (c) Copyright OCP-IP 2008
//  OCP-IP Confidential and Proprietary
//  $Id:

#ifndef _SlaveResponseQueue_h
#define _SlaveResponseQueue_h

#include "ocpip.h"

// Set to same value as Q-Slave (max requests outstanding)
// NOTE: would be more robust to move to deque
#define RESPONSE_QUEUE_DEPTH    256

// model a behavior response queue module with enqueue/dequeue events
template <typename TdataCl>
class ResponseQueue {
  public:
    // type definitions
    typedef typename TdataCl::DataType Td;
    typedef typename TdataCl::AddrType Ta;

  private:
    int      m_Head;
    int      m_Tail;
    sc_core::sc_event m_WaitOnDeQueueEvent;
    bool     m_WaitOnDeQueue;
    sc_core::sc_event m_WaitOnEnQueueEvent;
    bool     m_WaitOnEnQueue;
    bool     m_Reset;

    struct {
        ocpip_legacy::OCPResponseGrp<Td> Resp;
        sc_core::sc_time      SendTime;
        bool         Valid;
    } m_Buffer[RESPONSE_QUEUE_DEPTH];

  public:
    ResponseQueue()
      : m_Head(0),
        m_Tail(0),
        m_WaitOnDeQueue(false),
        m_WaitOnEnQueue(false),
        m_Reset(false)
    {
        resetState();
    }

    void startReset()
    {
        m_Reset = true;
        // trigger my events to break out of any deadlock
        m_WaitOnDeQueueEvent.notify();
        m_WaitOnEnQueueEvent.notify();
        resetState();
    }

    void endReset()
    { 
        resetState();
        m_Reset = false;
    }

    void resetState()
    {
        int i;

        for (i = 0; i < RESPONSE_QUEUE_DEPTH; i++) {
            m_Buffer[i].Valid = false;
            m_Buffer[i].Resp.SResp = ocpip_legacy::OCP_SRESP_NULL;
            m_Buffer[i].Resp.SData = 0;
        }
        m_Head =0;
        m_Tail =0;
    }

    void enqueueBlocking(const ocpip_legacy::OCPResponseGrp<Td>& resp, const sc_core::sc_time& send_time)
    {
        // check for full
        if (m_Buffer[m_Tail].Valid) {
            if (!m_WaitOnEnQueue) {
                // go to the enqueue waiting state
                m_WaitOnEnQueue = true;
                sc_core::wait(m_WaitOnEnQueueEvent);
                if (m_Reset) {
                    // We reset while waiting
                    return;
                }
                m_WaitOnEnQueue = false;
            } else {
                std::cerr << "ERROR: Cannot handle two processes wait to enqueue in SlaveResponseQueue."
                     << std::endl;
                sc_core::sc_stop();
            }
        }

        // insertion
        m_Buffer[m_Tail].Valid = true;
        m_Buffer[m_Tail].Resp = resp;
        m_Buffer[m_Tail].SendTime = send_time;

        m_Tail++;
        if (m_Tail >= RESPONSE_QUEUE_DEPTH) m_Tail = 0;

        // check for wake-up
        if (m_WaitOnDeQueue) {
            // wake up the wait-on-dequeue process
            m_WaitOnDeQueueEvent.notify(sc_core::SC_ZERO_TIME);
        }
    }

    void dequeueBlocking(ocpip_legacy::OCPResponseGrp<Td>& resp, sc_core::sc_time& send_time)
    {
        // check for empty
        if (!m_Buffer[m_Head].Valid) {
            if (!m_WaitOnDeQueue) {
                // go to the dequeue waiting state
                m_WaitOnDeQueue = true;
                sc_core::wait(m_WaitOnDeQueueEvent);
                if (m_Reset) {
                    // We reset while waiting
                    return;
                }
                m_WaitOnDeQueue = false;
            } else {
                std::cerr << "ERROR: Cannot handle two processes wait to dequeue in SlaveResponseQueue.h."
                     << std::endl;
                sc_core::sc_stop();
            }
        }

        // deletion
        m_Buffer[m_Head].Valid = false;
        resp = m_Buffer[m_Head].Resp;
        send_time = m_Buffer[m_Head].SendTime;

        m_Head++;
        if (m_Head >= RESPONSE_QUEUE_DEPTH) m_Head = 0;

        // check for wake-up
        if (m_WaitOnEnQueue) {
            // wake up the wait-on-enqueue process
            m_WaitOnEnQueueEvent.notify(sc_core::SC_ZERO_TIME);
        }
    }

    bool empty()
    {
       return !(m_Buffer[m_Head].Valid);
    }

    bool getHeadSResp(ocpip_legacy::OCPSRespType& headSResp)
    {
        if (empty()) {
            return false;
        }
        headSResp = m_Buffer[m_Head].Resp.SResp;
        return true;
    }

    bool getHeadSendTime(sc_core::sc_time& headSendTime)
    {
        if (empty()) {
            return false;
        }
        headSendTime = m_Buffer[m_Head].SendTime;
        return true;
    }

    int length()
    {
        // Is the queue empty?
        if (!m_Buffer[m_Head].Valid) {
            return 0;
        }
        int my_len = m_Tail - m_Head;
        if (my_len < 1) {
            my_len += RESPONSE_QUEUE_DEPTH;
        }
        return my_len;
    }

    void purgePlaceholders()
    {
        // Starting at the head of the list, removes any place holder
        // "dummy" posted write responses that have reached their time out
        
        // TODO: Note that it is possible that expired place holder "dummy"
        //       responses are sitting behind a regular response.
        //       For example, if the queue looked like this:
        //       head  --->  Read_Response
        //                   Place_Holder_Expired
        //                   Place_Holder_Expired
        //       tail  --->  Read_Response
        //       Then just looking at the head would miss the expired placeholders
        //       behind it. This circular buffer really needs to be a List or some
        //       other structure that would let us move through the queue and purge
        //       items in the middle if needed.
        
        // Keep removing expired place holders from the head:
        sc_core::sc_time CurTime = sc_core::sc_time_stamp();

        while ( (m_Buffer[m_Head].Valid) && 
                (m_Buffer[m_Head].Resp.SResp==ocpip_legacy::OCP_SRESP_NULL) && 
                (m_Buffer[m_Head].SendTime<=CurTime) ) {

            // This is a place holder - delete it.
            m_Buffer[m_Head].Valid = false;
            m_Head++;
            if (m_Head >= RESPONSE_QUEUE_DEPTH) {
                m_Head = 0;
            }
        }
    }
};

#endif // _SlaveResponseQueue_h
