// 
//  Copyright 2005 OCP-IP
//
// ============================================================================
//      Project : OCP SLD WG
//       Author : Anssi Haverinen, Nokia Inc.
//           $Id: slave.cc,v 1.1 2005/01/07 03:42:33 Anssi Exp $
//
//  Description : OCP API - TL1 profile example
//                Simple 2-port slave with no real arbitration
//                ReadEx is not implemented.
// ============================================================================

#include "slave.h"

// ----------------------------------------------------------------------------
// Constructor
// ----------------------------------------------------------------------------
Slave::Slave (sc_module_name name_)
  : sc_module (name_), wP("wPort"), rP("rPort") {

  // initialize common members
  is_req_w = false;
  is_req_r = false;
  request_w = false;
  request_r = false;
  not_accepted_w = false;
  not_accepted_r = false;
  address_w = 0;
  address_r = 0;
  burststart_w = true;
  burststart_r = true;
  burstlen_r = 0;
  burstlen_w = 0;
  bytemask = 0;
  for (int i=0;i<1024;i++)
    memory[i] = 0;

  SC_METHOD(proc_w);
  sensitive_pos(clk);
  dont_initialize(); 

  SC_METHOD(proc_r);
  sensitive_pos(clk);
  dont_initialize(); 


}


// ----------------------------------------------------------------------------
// Destructor
// ----------------------------------------------------------------------------
Slave::~Slave(){}


// ----------------------------------------------------------------------------
//  Method : Slave::proc_w()
//
//  Synchronous slave write port process
//
// ----------------------------------------------------------------------------

void Slave::proc_w(){

  bool tmp;

  // Request handshake
  is_req_w = wP->getOCPRequest(req_w);

  if (is_req_w)

#ifdef DEBUG_G1
    cout << "Slave got write request "
	 << " time " << sc_time_stamp().to_seconds()
	 << " MCmd " << req_w.MCmd
	 << " MAddr " << req_w.MAddr << endl;
#endif

  if (is_req_w || request_w) {
    if (!req_fifo_w.full()) {
      
      // Burst addressing
      if (burstlen_w == 0) {
	address_w = req_w.MAddr/4; // First of burst
	burstlen_w = req_w.MBurstLength;
      }
      else {
	if (req_w.MBurstSeq==OCP_MBURSTSEQ_INCR)
	  address_w = address_w+1; //incrementing address
	else if (req_w.MBurstSeq==OCP_MBURSTSEQ_STRM)
	  address_w = address_w; //streaming address
	else if (req_w.MBurstSeq==OCP_MBURSTSEQ_WRAP) {
	  if ((address_w+1) < ((address_w & (~(req_w.MBurstLength-1)))+ req_w.MBurstLength)) // wrapping address
	    address_w = address_w+1; 
	  else
	    address_w = address_w & (~(req_w.MBurstLength-1));
	}
      }

      burstlen_w--;
      request_w = false;
      if (burstlen_w>0) {
	if (req_w.MBurstSingleReq) {
	  request_w = true;
	  req_fifo_w.poke_flag(0); // For marking whether there needs to
	                           // be a response
	}
	else {
	  wP->putSCmdAccept();
	  req_fifo_w.poke_flag(1);
#ifdef DEBUG_G1
	  cout << "Slave accepted write request "
	       << " time " << sc_time_stamp().to_seconds() << endl;
#endif

	}
      }
      else {
	  wP->putSCmdAccept();
	  req_fifo_w.poke_flag(1);

#ifdef DEBUG_G1
      cout << "Slave accepted write request "
	   << " time " << sc_time_stamp().to_seconds() << endl;
#endif

      }
      req_fifo_w.write(address_w);
    }
    else
      request_w = true;
  }

  // Data handshake
  if (!req_fifo_w.empty() && !resp_fifo_w.full()) {
    is_data_w = wP->getOCPDataHS(req_data, true);
    if (is_data_w) {
      // Byte enable to bitmask
      if (req_data.MDataByteEn & 0x1)
	bytemask = 0xFF; 
      if  (req_data.MDataByteEn & 0x2)
	bytemask = 0xFF00 | bytemask; 
      if  (req_data.MDataByteEn & 0x4)
	bytemask = 0xFF0000 | bytemask; 
      if  (req_data.MDataByteEn & 0x8)
	bytemask = 0xFF000000 | bytemask; 

      memory[req_fifo_w.peek()] = req_data.MData & bytemask;
      if (req_fifo_w.peek_flag()) {
	resp_fifo_w.write(req_data.MData); // push response fifo
      }
      req_fifo_w.read();

#ifdef DEBUG_G1
      cout << "Slave got data "
	   << " time " << sc_time_stamp().to_seconds()
	   << " MData " << req_data.MData << endl;
#endif

    }
  }

  // Response handshake
  if (!resp_fifo_w.empty()) {
    tmp = wP->getMBusy();
    if (!tmp) {
      // Set OCP response
      resp_w.SResp = OCP_SRESP_DVA;
      // Send response
      wP->startOCPResponse(resp_w);

#ifdef DEBUG_G1
      cout << "Slave sent write response "
	   << " time " << sc_time_stamp().to_seconds()
	   << " SResp " << resp_w.SResp << endl;
#endif

      resp_fifo_w.read(); // pop response fifo
    }
  }
} // end of method

// ----------------------------------------------------------------------------
//  Method : Slave::proc_r()
//
//  Synchronous slave write port process
//
// ----------------------------------------------------------------------------

void Slave::proc_r(){
  bool tmp;

  // Request handshake
  is_req_r = rP->getOCPRequest(req_r);


#ifdef DEBUG_G1
  if (is_req_r)
      cout << "Slave got read request "
	   << " time " << sc_time_stamp().to_seconds()
	   << " MCmd " << req_r.MCmd
	   << " MBurstSingleReq " << req_r.MBurstSingleReq
	   << " MAddr " << req_r.MAddr << endl;
#endif

  if (is_req_r || request_r) {
    if (!req_fifo_r.full() && !resp_fifo_r.full()) {

      // Byte enable to bitmask
      if (req_r.MByteEn & 0x1)
	bytemask = 0xFF; 
      if  (req_r.MByteEn & 0x2)
	bytemask = 0xFF00 | bytemask; 
      if  (req_r.MByteEn & 0x4)
	bytemask = 0xFF0000 | bytemask; 
      if  (req_r.MByteEn & 0x8)
	bytemask = 0xFF000000 | bytemask; 
      
      // Burst addressing
      if (burstlen_r == 0) {
	address_r = req_r.MAddr/4; // First of burst
	burstlen_r = req_r.MBurstLength;
      }
      else {
	if (req_r.MBurstSeq==OCP_MBURSTSEQ_INCR)
	  address_r = address_r+1; //incrementing address
	else if (req_r.MBurstSeq==OCP_MBURSTSEQ_STRM)
	  address_r = address_r; //streaming address
	else if (req_r.MBurstSeq==OCP_MBURSTSEQ_WRAP) {
	  if ((address_r+1) < ((address_r & (~(req_r.MBurstLength-1)))+ req_r.MBurstLength)) // wrapping address
	    address_r = address_r+1; 
	  else
	    address_r = address_r & (~(req_r.MBurstLength-1));
	}
      }

      burstlen_r--;

      req_fifo_r.write(address_r);

      request_r = false;

      if (burstlen_r>0) {
	resp_fifo_r.write(0); // For SRespLast
	if (req_r.MBurstSingleReq)
	  request_r = true;
	else {
	  rP->putSCmdAccept();

#ifdef DEBUG_G1
      cout << "Slave accepted read request "
	   << " time " << sc_time_stamp().to_seconds() << endl;
#endif

	}
      }
      else {
	resp_fifo_r.write(1);
	rP->putSCmdAccept();

#ifdef DEBUG_G1
      cout << "Slave accepted read request "
	   << " time " << sc_time_stamp().to_seconds() << endl;
#endif

      }
      
    }
    else
      request_r = true;
  }


  

  // Response handshake with memory address arbitration
  // Same address cannot be read and written at the same time.
  // This does not always work correctly... Do not use...
  if (!req_fifo_r.empty() && (req_fifo_w.peek()!=req_fifo_r.peek())) {
    tmp = rP->getMBusy();
    if (!tmp) {
      // Set OCP response
      resp_r.SResp = OCP_SRESP_DVA;
      resp_r.SData = memory[req_fifo_r.read()];
      resp_r.SRespLast = resp_fifo_r.read();
      // Send response
      rP->startOCPResponse(resp_r);

#ifdef DEBUG_G1
      cout << "Slave sent read response "
	   << " time " << sc_time_stamp().to_seconds()
	   << " SData " << resp_r.SData 
	   << " SResp " << resp_r.SResp << endl;
#endif

    }
  }
} // end of method
