///////////////////////////
//
// Simple Main to read in Map data from files
// and then use that to configure and connect
// a master and slave.
//
///////////////////////////

#include <map>
#include <set>
#include <string>
#include <algorithm>
#include <stdio.h>
#include <stdlib.h>
#include <iostream>

#include "systemc.h"

#include "simpleMaster.h"
#include "simpleSlave.h"
#include "ocp_tl1_data_cl.h"
#include "ocp_tl_param_cl.h"
#include "ocp_tl1_channel_clocked.h"

#define OCP_CLOCK_PERIOD 1
#define OCP_CLOCK_TIME_UNIT SC_NS

void process_command_line(int   argc,
                          char* argv[],
                          string& ocp_params_file_name,
                          string& module_params_file_name,
                          double& simulation_end_time,
                          bool& debug_dump,
                          string& debug_file_name)
{
    // get the ocp parameters file name
    ocp_params_file_name = "";
    if (argc > 1) {
        string file_name(argv[1]);
        ocp_params_file_name = file_name;
    }

    // get the module parameters file name
    module_params_file_name = "";
    if (argc > 2) {
        string file_name(argv[2]);
        module_params_file_name = file_name;
    }

    // get the simulation end time
    simulation_end_time = 1000;
    if (argc > 3) {
        simulation_end_time = (double) atoll(argv[3]);
    }

    // do we dump out a log file?
    debug_dump= false;
    debug_file_name = "";
    if (argc > 4) {
        string file_name(argv[4]);
        debug_file_name = file_name;
        debug_dump = true;
    }
}

void readMapFromFile(const string &myFileName, MapStringType &myParamMap) 
{
    // read pairs of data from the passed file
    string leftside;
    string rightside;
    
    // (1) open the file
    ifstream inputfile(myFileName.c_str());
    assert( inputfile );

    // set the formatting
    inputfile.setf(std::ios::skipws);

    // Now read through all the pairs of values and add them to the passed map
    while ( inputfile ) {
        inputfile >> leftside;
        inputfile >> rightside;
        myParamMap.insert(std::make_pair(leftside,rightside));
    }

    // All done, close up
    inputfile.close();
}

int
sc_main(int argc, char* argv[])
{
    OCP_TL1_Channel_Clocked< OCP_TL1_DataCl<OCPCHANNELBit32, OCPCHANNELBit32> >* pOCP;
    Master< OCP_TL1_DataCl<OCPCHANNELBit32, OCPCHANNELBit32> >* pMaster;
    Slave< OCP_TL1_DataCl<OCPCHANNELBit32, OCPCHANNELBit32> >*  pSlave;
    MapStringType  ocpParamMap;
    MapStringType  moduleParamMap;

    double         simulation_end_time;
    bool           debug_dump;
    string         ocpParamFileName;
    string         moduleParamFileName;
    string         dump_file_name;
    ofstream       debugFile;

    // --------------------------------
    // (1) process command line options
    //     and read my parameters
    // --------------------------------
    process_command_line(argc,argv,ocpParamFileName,moduleParamFileName,
           simulation_end_time,debug_dump,dump_file_name);

    if ( ! ocpParamFileName.empty() ) {
        readMapFromFile(ocpParamFileName, ocpParamMap);
    }

    if ( ! moduleParamFileName.empty() ) {
        readMapFromFile(moduleParamFileName, moduleParamMap);
    }

    // open a trace file
    if (debug_dump) {
        cout << "Debug dumpfilename: " << dump_file_name << endl;
        debugFile.open(dump_file_name.c_str());
    }

    // ----------------------------------------------------------
    // (2) Create the clocked OCP Channel
    // ----------------------------------------------------------
    sc_clock clk("clk", OCP_CLOCK_PERIOD,OCP_CLOCK_TIME_UNIT) ;

    pOCP = new OCP_TL1_Channel_Clocked< OCP_TL1_DataCl<OCPCHANNELBit32, OCPCHANNELBit32> > ("ocp0");

    // ----------------------------------------------------------
    // (3) Create the Master and Slave
    // ----------------------------------------------------------

    pMaster = new Master< OCP_TL1_DataCl<OCPCHANNELBit32, OCPCHANNELBit32> >("master", 
                                                                              0, 
                                                                              &debugFile 
                                                                              );
    pSlave = new Slave< OCP_TL1_DataCl<OCPCHANNELBit32, OCPCHANNELBit32> >("slave", 
                                                                            0, 
                                                                            0x3FF,
                                                                            &debugFile 
                                                                            );
                                                                            
    pMaster->setModuleConfiguration(moduleParamMap);
    pSlave->setModuleConfiguration(moduleParamMap);

    // ----------------------------------------------------------
    // (4) connect channel, master, slave, and clock
    // ----------------------------------------------------------
    pMaster->ipP(*pOCP);
    pSlave->tpP(*pOCP);
    pMaster->clk(clk);
    pSlave->clk(clk);
    pOCP->p_clk(clk);

    // ------------------------
    // (5) start the simulation
    // ------------------------
    pMaster->provideChannelConfiguration(ocpParamMap);
    sc_start(simulation_end_time,SC_NS);

    // -------------------
    // (6) post processing
    // -------------------

    cout << "main program finished at " 
         << sc_time_stamp().to_double() << endl;

    sc_simcontext* sc_curr_simcontext = sc_get_curr_simcontext();
    cout << "delta_count: " << dec << sc_curr_simcontext->delta_count() 
             << endl;
    cout << "next_proc_id: " << dec << sc_curr_simcontext->next_proc_id()
         << endl;
         
    return (0);
}
