// 
//  (c) Copyright OCP-IP 2005
//  OCP-IP Confidential and Proprietary
//
// ============================================================================
//      Project : OCP SLD WG
//       Author : Tim Kogel, CoWare, Inc
//          $Id:
//
//  Description :OSCI TLM based implementation of the TL3 slave delayed interface
//
// ============================================================================

#ifndef _OCP_TL3_SLAVE_PROTOCOL_H
#define _OCP_TL3_SLAVE_PROTOCOL_H

#include "tlm.h"

#include "ChronoQueue.h"

#include "ocp_tl3_slave_if.h"
#include "ocp_tl3_monitor_if.h"

template 
<
  typename REQ,
  typename RESP
  >
class OCP_TL3_Slave_Protocol :
  public sc_module,
  virtual public OCP_TL3_SlaveIF<REQ,RESP>
{
private:
  typedef ChronoQueue<RESP,
		      MemoryPolicyPool,
		      QueueingPolicyOutOfOrder, 
		      NotifyPolicySCEvent> response_queue_type;


  SC_HAS_PROCESS(OCP_TL3_Slave_Protocol);

public:
  typedef tlm::tlm_nonblocking_get_peek_if<REQ> get_interface_type ; 
  typedef tlm::tlm_nonblocking_put_if<RESP> put_interface_type ; 
  typedef sc_port<get_interface_type> get_port_type;
  typedef sc_port<put_interface_type> put_port_type;

  typedef OCP_TL3_MonitorIF<REQ,RESP>			      monitor_if_type;
  typedef typename OCP_TL3_MonitorIF<REQ,RESP>::observer_type observer_type;

  get_port_type m_requestPort;
  put_port_type m_responsePort;

  //---------------------------------------------------------------
  // constructor
  //---------------------------------------------------------------
  OCP_TL3_Slave_Protocol(sc_module_name name, monitor_if_type* channel) :
    sc_module(name),
    m_requestPort("requestP"),
    m_responsePort("responseP"),
    m_requestUnread(true),
    m_response_queue(10),
    m_clkPeriod(1,SC_NS),
    m_channel(channel)
  { 
    SC_METHOD(accept_method);
    sensitive << m_acceptEvent;
    dont_initialize();

    SC_METHOD(response_method);
    sensitive << m_response_queue.getEvent();
    dont_initialize();
    
    SC_METHOD(notify_response_start_observer);
    sensitive << m_responseStartObserver;
    dont_initialize();    
  }

  //---------------------------------------------------------------
  // destructor
  //---------------------------------------------------------------
  virtual ~OCP_TL3_Slave_Protocol()
  { 
  }

  void end_of_elaboration() 
  {
    SC_METHOD(end_of_response_method);
    sensitive << ResponseEndEvent();
    dont_initialize();  
  }

public:
  //////////////////////////////////////////////////////////////
  // Map TL3 Slave Interface Methods to OSCI TLM API
  //////////////////////////////////////////////////////////////

  
  // TL3 slave request methods
  bool getRequest(REQ& req)
  { 
    // Check to see if there is already a transaction in progress
    // that we haven't read yet
    if (m_requestUnread && m_requestPort->nb_peek(req)) {
      // There is a transaction to get
      // Mark it as read
      m_requestUnread = false;
      return true;
    }
    return false;
  }

  bool getRequestBlocking(REQ& req)
  { 
    if (!getRequest(req)) {
      wait(m_requestPort->ok_to_peek());
      return getRequest(req);
    }
    return true;
  }

  bool acceptRequest()
  { return acceptRequest(SC_ZERO_TIME); }

  bool acceptRequest(const sc_time& accept_time)
  {
    if ( requestInProgress() ) {
      m_acceptEvent.notify(accept_time);
      return true;
    }
    return false;
  }

  bool acceptRequest(const int cycles)	     
  { 
    if ( requestInProgress() ) {
      if (cycles <= 0) {
	// Accept Right now
	return acceptRequest(SC_ZERO_TIME);
      } else {
	// wait for "cycles" clock periods and accept then
	return acceptRequest(cycles * m_clkPeriod);
      }
    }
    return false;
  }

  bool requestInProgress(void)  const 
  { return m_requestPort->nb_can_peek(); }

  const sc_event& RequestStartEvent(void) const 
  { return  m_requestPort->ok_to_peek(); }
  const sc_event& RequestEndEvent(void) const 
  { return m_requestEndEvent; }


  // TL3 slave response methods
  bool sendResponse(const RESP& resp)
  { return sendResponse(resp,SC_ZERO_TIME); }

  bool sendResponse(const RESP& resp, const sc_time& time)
  {
    m_response_queue.createNew();
    m_response_queue.getNewItem() = resp;
    m_response_queue.getNewSendTime() = time + sc_time_stamp();
    m_response_queue.enqueueNew();
    return true;
  }

  bool sendResponse(const RESP& resp, const int cycles)
  { return sendResponse(resp, cycles * m_clkPeriod); }
  
  bool sendResponseBlocking(const RESP& resp)
  {  
    if (responseInProgress()) {
      wait(ResponseEndEvent());
    }
    if (!sendResponse(resp)) {
      return false;
    }
    wait(ResponseEndEvent());
    return true;
  }

  bool responseInProgress(void) const
  { return !m_responsePort->nb_can_put(); }
    
  // Event access
  const sc_event& ResponseStartEvent(void) const
  { return m_responseStartEvent; }
  const sc_event& ResponseEndEvent(void) const
  { return m_responsePort->ok_to_put(); }
  
  void setPeriod( const sc_time& newPeriod )
  {
    m_clkPeriod = newPeriod;
  }

  const sc_time& getPeriod(void) const 
  {
    return m_clkPeriod;
  }

  void RegisterRequestEnd(observer_type* monitor)
  { m_request_end_observer.push_back(monitor); }
  void RegisterResponseStart(observer_type* monitor)
  { m_response_start_observer.push_back(monitor); }

private:
  void response_method()
  {
#ifndef NDEBUG
    if ( !m_response_queue.nextItemDue() ) {
      cout << name() << ":\n\nERROR response_method! next send time " 
	   <<  m_response_queue.peekNextSendTime() << " not yet due" << endl;
      return;
    }

    if ( responseInProgress() ) {
      // this should never happen! the activation scheme is designed
      // so that when this method notified, the response channel should always be free.
      cerr << name() << ":\n\nERROR response_method! responseInProgress should never be true!";
      return;
    }
    //  cout << sc_time_stamp() << ", " << name() << ": send response\n";
#endif
  
    if ( m_responsePort->nb_can_put() ) {
      // Put the transaction on the channel
      m_responsePort->nb_put(m_response_queue.peekNextItem());
      
      // Trigger the event
      m_responseStartEvent.notify();
      m_responseStartObserver.notify(SC_ZERO_TIME);
    } else {
      // Already a transaction in progress. Failed.
      cerr << name() << ":\n\nERROR response_method! put response failed" << endl;
    }
  }

  void end_of_response_method()
  {
    //    cout << sc_time_stamp() << ", " << name() << ": response ends\n";
    m_response_queue.dequeueAndDestroy();
  }

  void accept_method(void)
  {
    if ( requestInProgress() ) {
      m_requestUnread = true;
      m_requestEndEvent.notify();
      m_requestPort->nb_get(m_req);
      notify_request_end_observer();
    } else {
      cerr << name() << ":\n\nERROR accept_method! responseInProgress is be false!";
    }
  }

    void notify_request_end_observer()
    {
      for( observer_iterator_type
	     it  = m_request_end_observer.begin(),
	     end = m_request_end_observer.end();
	   it != end;
	   ++it) {
	(*it)->NotifyRequestEnd( m_channel );
      }
    }
    void notify_response_start_observer()
    {
      for( observer_iterator_type
	     it  = m_response_start_observer.begin(),
	     end = m_response_start_observer.end();
	   it != end;
	   ++it) {
	(*it)->NotifyResponseStart( m_channel );
      }
    }

  //----------------------------------------------------------------------
  // member data
  //----------------------------------------------------------------------
  sc_event m_requestEndEvent;
  sc_event m_responseStartEvent;
  bool m_requestUnread;
  REQ m_req;
  response_queue_type m_response_queue;
  sc_time m_clkPeriod;
  sc_event m_acceptEvent;
  sc_event m_responseStartObserver;

  typedef ::std::vector< observer_type* > observer_container_type;
  typedef typename observer_container_type::iterator observer_iterator_type;
  observer_container_type m_request_end_observer;
  observer_container_type m_response_start_observer;
  monitor_if_type* m_channel;
};

#endif  // _TLM_TL3_SLAVE_PROTOCOL_H
