// 
//  Copyright 2003 OCP-IP
//
// ============================================================================
//      Project : OCP SLD WG, OCP Transaction Level Layer-1
//       Author : Norman Weyrich, Synopsys Inc., weyrich@synopsys.com
//                Anssi Haverinen, Nokia Inc., anssi.haverinen@nokia.com
//         Date : 02/15/2003
//
//  Description : Transaction Level - Layer-1 example Slave
//    Features:
//    - Synchronous slave using OCP data class
//    - Non-blocking methods are used
//    - Pipelined Request and Response threads
//    - Use additional sampling thread (alternative to the usage of pre-emptive
//      request channel release method SrelesePE()
//                
// Parameter    :
//   Template arguments.
//     - TdataCl: Data class containing data members and
//                access methods for the data exchanged between
//                Masters and Slaves
//     - Td: Data type
//     - Ta: Address type
//                
//   Constructor arguments.
//     - sc_module_name: Name of the module instance
//     - ID: Unique number identifying the SLave.
//           ID must be unique among all Slaves attached to the same Bus.
//     - StartAddress: First address of the Slaves memory.
//     - EndAddress: Last address of the Slaves memory.
//                   The address range of all Slaves in a system
//                   must be mutually distinct.
//
//
// ============================================================================

#include "ocp_tl1_slave_sync1.h"
#include "ocp_tl1_data_cl.h"

// ----------------------------------------------------------------------------
// Process : OCP_TL1_Slave_Sync::OCP_TL1_Slave_Sync
// 
// Constructor
// ----------------------------------------------------------------------------
template<class TdataCl> OCP_TL1_Slave_Sync<TdataCl>::OCP_TL1_Slave_Sync
       (
          sc_module_name name_
        , int ID
        , Ta StartAddress
        , Ta EndAddress
        )
          : sc_module          (name_)
          , SlaveP             ("SlavePort")
          , m_ID               (ID)
          , m_StartAddress     (StartAddress)
          , m_EndAddress       (EndAddress)
{
  int i;

  // InitParameters();

  m_NumWordsPerBeat = sizeof(Td);
  m_SmemoryLength = (m_EndAddress - m_StartAddress + 1) / m_NumWordsPerBeat;

  // allcoate memory
  if (m_SmemoryLength > 0)
  {
    try
    {
      m_SlaveMemory = new Td[m_SmemoryLength];
    }
    catch(std::bad_alloc ex)
    {
      cerr << "ERROR: Memory allocation for Slave memory failed "
            << name() << ">" << endl;
      cerr << "       Size  is: " << m_SmemoryLength << endl;
      cerr << "       Error is: " << ex.what() << endl;
      sc_stop();
      return;
    }
  }

  for (i = 0; i < m_SmemoryLength; i++)
    m_SlaveMemory[i] = 0;

  for (i = 0; i < TL_SLAVE_FIFO_DEPTH; i++)
    m_RespEventS[i] = false;

  m_Pause = 0;

  SC_THREAD(Release);
  sensitive << SlaveP;
  SC_THREAD(Request);
  sensitive_pos(clk);
  SC_THREAD(Response);
  sensitive << m_RespEvent;
}


// ----------------------------------------------------------------------------
// Process : OCP_TL1_Slave_Sync::~OCP_TL1_Slave_Sync
// 
// Destructor
// ----------------------------------------------------------------------------
template<class TdataCl> OCP_TL1_Slave_Sync<TdataCl>::~OCP_TL1_Slave_Sync()
{
  delete [] m_SlaveMemory;
}


// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Slave_Sync::Release()
//
//  Top level asynchronous request sampling
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL1_Slave_Sync<TdataCl>::Release()
{
  while(true)
  {
    wait(); // for default event

    if (m_Pause == 0)
      SlaveP->SgetRequest(true);
    else
    {
      SlaveP->SgetRequest(false);
      SlaveP->Srelease(sc_time(TL_CLK_PERIOD * m_Pause, TL_TIME_UNIT));
    }
  } // end of while(true) loop
} // end of method


// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Slave_Sync::Request()
//
//  Top level synchronous data sampling method
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL1_Slave_Sync<TdataCl>::Request()
{
  Td Data;
  Ta Addr;
  OCPMCmdType MCmd;

  int i = 0;
  int ReqIdx = 0;
  int Pause[] = {0, 0, 0, 0, 0, 1, 1, 1, 2, 3};
//  int Pause[] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

  while(true)
  {
    wait(); // for rising clock edge

    MCmd = m_DataCl->SgetMCmd(); // check if a request was posted
    if (MCmd != OCP_MCMD_IDLE)
    {
      i++;
      m_DataCl->SgetMData(Data);
      Addr = m_DataCl->SgetMAddr();
      
      if (MCmd == OCP_MCMD_RD)
      {
        // send response in same delta cycle
        m_RespEventS[ReqIdx] = true;
        m_RespEvent.notify();

        // get new index (only for those requests, where the
        // response thread is called)
        ReqIdx++;
        if (ReqIdx >= TL_SLAVE_FIFO_DEPTH)
          ReqIdx = 0;
      }

#ifdef DEBUG_G1
      cout << "Slave got request "
           << "  delta = " << simcontext()->delta_count()
           << "  time  = " << sc_time_stamp().to_seconds()
	   << "  data  = " << Data << endl;
#endif
      m_Pause = Pause[i%10];
    }
  } // end of while(true) loop
} // end of method


// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Slave_Sync::Response()
//
//  Top level response method (synchronious)
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL1_Slave_Sync<TdataCl>::Response()
{
  Td Rdata = 0;
  int ResIdx = 0;

  while(true)
  {
    if (!m_RespEventS[ResIdx])
      wait(); // for m_RespEvent event

    m_DataCl->SputSResp(OCP_SRESP_DVA); // Toggles the response data members
    m_DataCl->SputSData(Rdata++);

    // reset + get new index
    m_RespEventS[ResIdx] = false;
    ResIdx++;
    if (ResIdx >= TL_SLAVE_FIFO_DEPTH)
      ResIdx = 0;

    // send response to Master
    while (!SlaveP->SputResponse())
      wait(clk->posedge_event());
  }
}


// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Slave_Sync::end_of_elaboration()
//
//  This method is activated at the end of the elaboration phase
//  (when all binding operations are finished).
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL1_Slave_Sync<TdataCl>::end_of_elaboration()
{
  sc_module::end_of_elaboration();

  // Initialize debug interface
  SlaveP->SregisterDirectIF(this);

  // Get data structure
  m_DataCl = SlaveP->GetDataCl();

  // Get communication structure
  m_CommCl = SlaveP->GetCommCl();

  // Get system parameter structure
  m_ParamCl = SlaveP->GetParamCl();

  // Put parameter into Channel
  // No parameters needed
}


// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Slave_Sync::MPutDirect()
//
//  Debug interface method.
//  Read/Write data, Master to Slave direction
//  Returns true for success and false for failure
//
// ----------------------------------------------------------------------------
template<class TdataCl> bool OCP_TL1_Slave_Sync<TdataCl>::MputDirect(
         int MasterID, bool IsWrite, Td *Data, Ta Address, int NumWords)
{
  int i;
  int NumBeats = NumWords / m_NumWordsPerBeat;
  int Offset = (Address - m_StartAddress) / m_NumWordsPerBeat;

  if ((Offset >= 0) && ((Offset + NumBeats) <= m_SmemoryLength))
  {
    if (IsWrite)
    {
      // write request
      for (i = 0; i < NumBeats; i++)
        m_SlaveMemory[Offset + i] = Data[i];
    }
    else
    {
      // read request
      for (i = 0; i < NumBeats; i++)
        Data[i] = m_SlaveMemory[Offset + i];
    }
    return(true);
  }
  else
  {
    cout << "Error: Address out of range at Slave " << m_ID << endl
         << "       Address=" << Address << "  StartAddr=" << m_StartAddress
         << "  _EndAddr=" << m_EndAddress
         << "  NumWords=" << NumWords
         << endl;
    return(false);
  }
}


// ----------------------------------------------------------------------------
//
//  Instantiation of the Slave
//
// ----------------------------------------------------------------------------
template class OCP_TL1_Slave_Sync<TL1_TEMPL_DATA_CL >; // see ocp_tl1_globals.h
