///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright 2006 OCP-IP
// OCP-IP Confidential & Proprietary
//
//
//============================================================================
//      Project : OCP SLD WG
//       Author : Tim Kogel, CoWare, Inc.
//          Date: 02/22/2006
//
//  Description :  OCP Channel Transaction Recording Monitor
//	  This channel monitor is based on the transaction recording 
//	  API in the SystemC Verification (SCV) Library. It targets 
//	  performance analysis for architectural modeling.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef _OCP_TL3_PERF_MONITOR_SCV_H
#define _OCP_TL3_PERF_MONITOR_SCV_H


#include "ocp_tl3_monitor_observer_if.h"
#include "ocp_perf_monitor_extensions.h"
#include "scv.h"

template <class, class> class OCP_TL3_MonitorIF;

template 
< typename REQ, 
  typename RESP
>
class OCP_TL3_Perf_Monitor_SCV: 
  public OCP_TL3_Monitor_ObserverIF<REQ,RESP>
{
public:
  typedef typename OCP_TL3_Monitor_ObserverIF<REQ,RESP>::tl3_peek_type tl3_peek_type;

  OCP_TL3_Perf_Monitor_SCV(const char* name); 
  virtual ~OCP_TL3_Perf_Monitor_SCV(); 

  virtual void NotifyRequestStart (tl3_peek_type *);
  virtual void NotifyRequestEnd   (tl3_peek_type *);
  virtual void NotifyResponseStart(tl3_peek_type *);
  virtual void NotifyResponseEnd  (tl3_peek_type *);

  virtual void registerChannel(tl3_peek_type *, 
			       bool master_is_node = false,
			       bool slave_is_node = false);

protected:
  scv_tr_stream*		m_req_stream;
  scv_tr_stream*		m_resp_stream;
  scv_tr_generator<>*		m_req_gen;
  scv_tr_generator<>*		m_resp_gen;

  scv_tr_handle			m_req_handle;
  scv_tr_handle			m_resp_handle;

  const char*			m_name;
  const char*			m_master_name;
  const char*			m_slave_name;
  bool				m_monitor_registered;
};

template 
< typename REQ, 
  typename RESP
>
OCP_TL3_Perf_Monitor_SCV<REQ,RESP>::OCP_TL3_Perf_Monitor_SCV(const char* name):
  m_name(name),
  m_master_name(NULL),
  m_slave_name(NULL),
  m_monitor_registered(false)
{
  m_req_stream = new scv_tr_stream(m_name,"request");
  m_resp_stream = new scv_tr_stream(m_name,"response");
  m_req_gen = new scv_tr_generator<>("request",*m_req_stream);
  m_resp_gen = new scv_tr_generator<>("response",*m_resp_stream);
  
}

template 
< typename REQ, 
  typename RESP
>
OCP_TL3_Perf_Monitor_SCV<REQ,RESP>::~OCP_TL3_Perf_Monitor_SCV()
{
  // delete stuff
  delete m_req_stream;
  delete m_resp_stream;
  delete m_req_gen;
  delete m_resp_gen;
}

template 
< typename REQ, 
  typename RESP
>
void
OCP_TL3_Perf_Monitor_SCV<REQ,RESP>::registerChannel(tl3_peek_type *tl3_channel, 
						    bool master_is_node,
						    bool slave_is_node)
{
  if (m_monitor_registered) {
    assert(m_master_name);
    assert(m_slave_name);
    cerr << "monitor " << m_name << " is allready registered:" << endl
	 << "master port name :  " << m_master_name << endl
	 << "slave port name :  " << m_slave_name << endl;
    return;
  }
  m_monitor_registered = true;
  m_master_name = tl3_channel->peekMasterPortName().c_str();
  m_slave_name =  tl3_channel->peekSlavePortName().c_str();
}

template 
< typename REQ, 
  typename RESP
>
void
OCP_TL3_Perf_Monitor_SCV<REQ,RESP>::NotifyRequestStart(tl3_peek_type *tl3_channel)
{
  m_req_handle = m_req_gen->begin_transaction();
}

template 
< typename REQ, 
  typename RESP
>
void 
OCP_TL3_Perf_Monitor_SCV<REQ,RESP>::NotifyRequestEnd(tl3_peek_type *tl3_channel)
{
  m_req_handle.end_transaction();
}

template 
< typename REQ, 
  typename RESP
>
void 
OCP_TL3_Perf_Monitor_SCV<REQ,RESP>::NotifyResponseStart(tl3_peek_type *tl3_channel)
{
  m_resp_handle = m_resp_gen->begin_transaction();
}

template 
< typename REQ, 
  typename RESP
>
void 
OCP_TL3_Perf_Monitor_SCV<REQ,RESP>::NotifyResponseEnd(tl3_peek_type *tl3_channel)
{
  m_resp_handle.end_transaction();
}

#endif
